---------------------------------------------------------------------------------------------------
--! @brief  Dual port RAM test bench
--! @details   The test bench simulates the module with various input signals value,
--! lenghts and timing.
--!
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date January 22 2018 created
--! @date March 29 2018 last modify
--!
--! @version v1.0
--!
--! @par Modifications:
--! jkokalj, January 22 2018: Created
--! jkokalj, March 28 2018: design iteration
--!
--! @file CslDpRAMTb.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslStdRtlPkg.all;

--! @brief  Dual port RAM test bench
--! @details   The test bench simulates the module with various input signals value,
--! lenghts and timing.
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
---------------------------------------------------------------------------------------------------
entity CslDpRAMTb is
   generic(
      TPD_G          : time     := 1 ns;
      ADR_W_G        : integer := 4;     --! addresa width
      DAT_W_G        : integer := 8 ;    --! data width
      PIPELINE_G     : natural := 1;     --! delay in clock cycles of output data
      WRITE_FIRST_G  : boolean := false; --! true=> write first mode, false=> read first mode
      BRAM_G         : boolean := true   --! false => LUT RAM, true => BRAM
    );
end CslDpRAMTb;
---------------------------------------------------------------------------------------------------
architecture behavior of CslDpRAMTb is 
 
   --Inputs
   signal clka : std_logic := '0';
   signal rsta : std_logic := '0';
   signal wea : std_logic := '0';
   signal addra : std_logic_vector(ADR_W_G-1 downto 0) := (others => '0');
   signal dina : std_logic_vector(DAT_W_G-1 downto 0) := (others => '0');
   signal clkb : std_logic := '0';
   signal rstb : std_logic := '0';
   signal addrb : std_logic_vector(ADR_W_G-1 downto 0) := (others => '0');

 	--Outputs
   signal douta_B_WR : std_logic_vector(DAT_W_G-1 downto 0);
   signal doutb_B_WR : std_logic_vector(DAT_W_G-1 downto 0);
   signal douta_D_WR : std_logic_vector(DAT_W_G-1 downto 0);
   signal doutb_D_WR : std_logic_vector(DAT_W_G-1 downto 0);
   signal douta_B_RD : std_logic_vector(DAT_W_G-1 downto 0);
   signal doutb_B_RD : std_logic_vector(DAT_W_G-1 downto 0);
   signal douta_D_RD : std_logic_vector(DAT_W_G-1 downto 0);
   signal doutb_D_RD : std_logic_vector(DAT_W_G-1 downto 0);
   signal douta_B_WR0 : std_logic_vector(DAT_W_G-1 downto 0);
   signal doutb_B_WR0 : std_logic_vector(DAT_W_G-1 downto 0);
   signal douta_D_WR0 : std_logic_vector(DAT_W_G-1 downto 0);
   signal doutb_D_WR0 : std_logic_vector(DAT_W_G-1 downto 0);
   signal douta_B_RD0 : std_logic_vector(DAT_W_G-1 downto 0);
   signal doutb_B_RD0 : std_logic_vector(DAT_W_G-1 downto 0);
   signal douta_D_RD0 : std_logic_vector(DAT_W_G-1 downto 0);
   signal doutb_D_RD0 : std_logic_vector(DAT_W_G-1 downto 0);

   -- Clock period definitions
   constant CLKA_I_PERIOD_C : time := 10 ns;
   constant CLKB_I_PERIOD_C : time := 7 ns;
--------------------------------------------------------------------------------------------------- 
BEGIN
 
	-- Instantiate the Unit Under Test (UUT)
   uut: entity work.CslDpRAM 
      generic map(
         TPD_G          =>TPD_G,
         ADR_W_G        =>ADR_W_G,       --! addresa width
         DAT_W_G        =>DAT_W_G,       --! data width
         PIPELINE_G     => 1,            --! delay in clock cycles of output data
         WRITE_FIRST_G  => true,         --! true=> write first mode, false=> read first mode
         BRAM_G         => true
      )
      PORT MAP (
         clka => clka,
         rsta => rsta,
         wea => wea,
         addra => addra,
         dina => dina,
         douta => douta_B_WR,
         clkb => clkb,
         rstb => rstb,
         addrb => addrb,
         doutb => doutb_B_WR
      );

	-- Instantiate the Unit Under Test (UUT)
   uut1: entity work.CslDpRAM 
      generic map(
         TPD_G          =>TPD_G,
         ADR_W_G        =>ADR_W_G,           --! addresa width
         DAT_W_G        =>DAT_W_G,           --! data width
         PIPELINE_G     => 1,                --! delay in clock cycles of output data
         WRITE_FIRST_G  => true,             --! true=> write first mode, false=> read first mode
         BRAM_G         => false
      )
      PORT MAP (
         clka => clka,
         rsta => rsta,
         wea => wea,
         addra => addra,
         dina => dina,
         douta => douta_D_WR,
         clkb => clkb,
         rstb => rstb,
         addrb => addrb,
         doutb => doutb_D_WR
      );
      
	-- Instantiate the Unit Under Test (UUT)
   uut2: entity work.CslDpRAM 
      generic map(
         TPD_G          =>TPD_G,
         ADR_W_G        =>ADR_W_G,           --! addresa width
         DAT_W_G        =>DAT_W_G,           --! data width
         PIPELINE_G     => 1,            --! delay in clock cycles of output data
         WRITE_FIRST_G  => false,         --! true=> write first mode, false=> read first mode
         BRAM_G         => true
      )
      PORT MAP (
         clka => clka,
         rsta => rsta,
         wea => wea,
         addra => addra,
         dina => dina,
         douta => douta_B_RD,
         clkb => clkb,
         rstb => rstb,
         addrb => addrb,
         doutb => doutb_B_RD
      );

	-- Instantiate the Unit Under Test (UUT)
   uut3: entity work.CslDpRAM 
      generic map(
         TPD_G          =>TPD_G,
         ADR_W_G        =>ADR_W_G,           --! addresa width
         DAT_W_G        =>DAT_W_G,           --! data width
         PIPELINE_G     => 1,                --! delay in clock cycles of output data
         WRITE_FIRST_G  => false,            --! true=> write first mode, false=> read first mode
         BRAM_G         => false
      )
      PORT MAP (
         clka => clka,
         rsta => rsta,
         wea => wea,
         addra => addra,
         dina => dina,
         douta => douta_D_RD,
         clkb => clkb,
         rstb => rstb,
         addrb => addrb,
         doutb => doutb_D_RD
      );
      
	-- Instantiate the Unit Under Test (UUT)
   uut4: entity work.CslDpRAM 
      generic map(
         TPD_G          =>TPD_G,
         ADR_W_G        =>ADR_W_G,           --! addresa width
         DAT_W_G        =>DAT_W_G,           --! data width
         PIPELINE_G     => 0,                --! delay in clock cycles of output data
         WRITE_FIRST_G  => true,             --! true=> write first mode, false=> read first mode
         BRAM_G         => true
      )
      PORT MAP (
         clka => clka,
         rsta => rsta,
         wea => wea,
         addra => addra,
         dina => dina,
         douta => douta_B_WR0,
         clkb => clkb,
         rstb => rstb,
         addrb => addrb,
         doutb => doutb_B_WR0
      );

	-- Instantiate the Unit Under Test (UUT)
   uut5: entity work.CslDpRAM 
      generic map(
         TPD_G          =>TPD_G,
         ADR_W_G        =>ADR_W_G,           --! addresa width
         DAT_W_G        =>DAT_W_G,           --! data width
         PIPELINE_G     => 0,                --! delay in clock cycles of output data
         WRITE_FIRST_G  => true,             --! true=> write first mode, false=> read first mode
         BRAM_G         => false
      )
      PORT MAP (
         clka => clka,
         rsta => rsta,
         wea => wea,
         addra => addra,
         dina => dina,
         douta => douta_D_WR0,
         clkb => clkb,
         rstb => rstb,
         addrb => addrb,
         doutb => doutb_D_WR0
      );
      
	-- Instantiate the Unit Under Test (UUT)
   uut6: entity work.CslDpRAM 
      generic map(
         TPD_G          =>TPD_G,
         ADR_W_G        =>ADR_W_G,           --! addresa width
         DAT_W_G        =>DAT_W_G,           --! data width
         PIPELINE_G     => 0,                --! delay in clock cycles of output data
         WRITE_FIRST_G  => false,            --! true=> write first mode, false=> read first mode
         BRAM_G         => true
      )
      PORT MAP (
         clka => clka,
         rsta => rsta,
         wea => wea,
         addra => addra,
         dina => dina,
         douta => douta_B_RD0,
         clkb => clkb,
         rstb => rstb,
         addrb => addrb,
         doutb => doutb_B_RD0
      );

	-- Instantiate the Unit Under Test (UUT)
   uut7: entity work.CslDpRAM 
      generic map(
         TPD_G          =>TPD_G,
         ADR_W_G        =>ADR_W_G,           --! addresa width
         DAT_W_G        =>DAT_W_G,           --! data width
         PIPELINE_G     => 0,                --! delay in clock cycles of output data
         WRITE_FIRST_G  => false,            --! true=> write first mode, false=> read first mode
         BRAM_G         => false
      )
      PORT MAP (
         clka => clka,
         rsta => rsta,
         wea => wea,
         addra => addra,
         dina => dina,
         douta => douta_D_RD0,
         clkb => clkb,
         rstb => rstb,
         addrb => addrb,
         doutb => doutb_D_RD0
      );
      
   --! @brief Clock A procces 
   --! @details Clock signal generator
   --! @param[in]  CLKA_I_PERIOD_C
   --! @param[out] clka
   p_ClkAGen :process
   begin
		clka <= '0';
		wait for CLKA_I_PERIOD_C/2;
		clka <= '1';
		wait for CLKA_I_PERIOD_C/2;
   end process;

   --! @brief Clock B procces 
   --! @details Clock signal generator
   --! @param[in]  CLKB_I_PERIOD_C
   --! @param[out] clkb 
   p_ClkBGen :process
   begin
		clkb <= '0';
		wait for CLKB_I_PERIOD_C/2;
		clkb <= '1';
		wait for CLKB_I_PERIOD_C/2;
   end process;
   
   --! @brief Simulation procces 
   --! @details Simulation with various signals value, length and timings
   --! @param[in] 
   --! @param[out] addra, dina, wea
   p_SimA: process
   begin		
   wait for CLKA_I_PERIOD_C;
      addra <="0001";
      dina  <="00001111";
      wea   <= '1';
      wait for CLKA_I_PERIOD_C;
      addra <="0010";
      dina  <="00001111";
      wea   <= '0';
      wait for CLKA_I_PERIOD_C;
      addra <="0011";
      dina  <="00011110";
      wea   <= '1';
      wait for CLKA_I_PERIOD_C;
      addra <="0100";
      dina  <="00111100";
      wea   <= '0';
      wait for CLKA_I_PERIOD_C;
      addra <="0010";
      dina  <="01111000";
      wea   <= '1';
      wait for CLKA_I_PERIOD_C;
      addra <="0011";
      dina  <="11110000";
      wea   <= '0';
      wait for CLKA_I_PERIOD_C;
      addra <="0100";
      dina  <="11100000";
      wea   <= '1';
      wait for CLKA_I_PERIOD_C;
      addra <="0001";
      dina  <="11000000";
      wea   <= '0';
      wait for CLKA_I_PERIOD_C;
      addra <="0001";
      dina  <="00001111";
      wea   <= '0';
      wait for CLKA_I_PERIOD_C;
      addra <="0010";
      dina  <="00001111";
      wea   <= '1';
      wait for CLKA_I_PERIOD_C;
      addra <="0011";
      dina  <="00011110";
      wea   <= '0';
      wait for CLKA_I_PERIOD_C;
      addra <="0100";
      dina  <="00111100";
      wea   <= '1';
      wait for CLKA_I_PERIOD_C;
      addra <="0010";
      dina  <="01111000";
      wea   <= '0';
      wait for CLKA_I_PERIOD_C;
      addra <="0011";
      dina  <="11110000";
      wea   <= '1';
      wait for CLKA_I_PERIOD_C;
      addra <="0100";
      dina  <="11100000";
      wea   <= '0';
      wait for CLKA_I_PERIOD_C;
      addra <="0001";
      dina  <="11000000";
      wea   <= '1';
      wait for CLKA_I_PERIOD_C;
      addra <="0011";
      dina  <="11000000";
      wea   <= '0';
      wait for CLKA_I_PERIOD_C*2;
   end process;

   --! @brief Simulation procces 
   --! @details Simulation with various signals value, length and timings
   --! @param[in] 
   --! @param[out] addrb
   p_SimB: process
   begin		
      wait for CLKB_I_PERIOD_C;
      addrb <="0001";
      wait for CLKB_I_PERIOD_C;
      addrb <="0010";
      wait for CLKB_I_PERIOD_C;
      addrb <="0011";
      wait for CLKB_I_PERIOD_C;
      addrb <="0100";
      wait for CLKB_I_PERIOD_C;
      addrb <="0010";
      wait for CLKB_I_PERIOD_C;
      addrb <="0011";
      wait for CLKB_I_PERIOD_C;
      addrb <="0100";
      wait for CLKB_I_PERIOD_C;
      addrb <="0001";
      wait for CLKB_I_PERIOD_C;
      addrb <="0001";
      wait for CLKB_I_PERIOD_C;
      addrb <="0010";
      wait for CLKB_I_PERIOD_C;
      addrb <="0011";
      wait for CLKB_I_PERIOD_C;
      addrb <="0100";
      wait for CLKB_I_PERIOD_C;
      addrb <="0010";
      wait for CLKB_I_PERIOD_C;
      addrb <="0011";
      wait for CLKB_I_PERIOD_C;
      addrb <="0100";
      wait for CLKB_I_PERIOD_C;
      addrb <="0001";
      wait for CLKB_I_PERIOD_C;
      addrb <="0011";
      wait for CLKB_I_PERIOD_C*2;
   end process;
end;
---------------------------------------------------------------------------------------------------