---------------------------------------------------------------------------------------------------
--! @brief     CMUX device 
--! @details
--!     Top module of the CMUX device. The CMUX interfaces to the first 28 SFP ports of the Arista DCS-7130-48L. 
--!     The 28 ports are linked to the ITU, FEC, 12 SFEDs, 6 DDSs, and 6 ITMs respectively.
--! 
--!     The SFP ports are connected to the GTY high-speed serial transceivers on the FPGA. 
--!        
--!     To drive the 28 GTY ports four GT reference clocks are required. 
--!     The REFCLK 1 of banks 227,229, 231 and 233 are used. The frequency of reference clocks is 125 MHz.
--!       
--! @author Uros Legat, Cosylab (uros.legat@cosylab.com)
--!
--! @date Oct 07 2019 created
--! @date Dec 20 2019 last modify
--!
--! @version v1.0
--!
--! @file CmuxDevice.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
use work.CslStdRtlPkg.all;
use work.FiberPkg.all;
--
use work.CmuCorePkg.all;

entity CmuxDevice is
   generic (
      TPD_G        : time := 1 ns;
      BUILD_INFO_G : BuildInfoType := (others => '0')
   );
   port (
      userClkP_i  : in  sl;
      userClkN_i  : in  sl;
      gtRefClkP_i : in  slv(3 downto 0);
      gtRefClkN_i : in  slv(3 downto 0);
      gtRxP_i     : in  slv(27 downto 0);
      gtRxN_i     : in  slv(27 downto 0);
      gtTxP_o     : out slv(27 downto 0);
      gtTxN_o     : out slv(27 downto 0)
   );
end CmuxDevice;

architecture structure of CmuxDevice is

   -- GT clocks
   signal gtFreeRst : sl;
   signal gtFreeClk : sl;
   signal gtRefClk  : slv(3 downto 0);
   -- CMU System clock 
   signal clk : sl;
   signal rst : sl;
   --! GT Data 
   signal gtRx      : GtArray(27 downto 0) := (others => GT_INIT_C);
   signal gtTx      : GtArray(27 downto 0) := (others => GT_INIT_C);
   signal gtRxRemap : GtArray(27 downto 0) := (others => GT_INIT_C);
   signal gtTxRemap : GtArray(27 downto 0) := (others => GT_INIT_C);
--------------
begin

   -----------------------------
   --! Generation of internal clocks and resets 
   -----------------------------
   u0_CmuClkRst : entity work.CmuClkRst
      generic map (
         TPD_G       => TPD_G,
         HOLD_TIME_G => RST_HOLD_TIME_C)
      port map (
         userClkP_i  => userClkP_i,
         userClkN_i  => userClkN_i,
         gtRefClkP_i => gtRefClkP_i,
         gtRefClkN_i => gtRefClkN_i,
         gtRefClk_o  => gtRefClk,
         gtFreeClk_o => gtFreeClk,
         gtFreeRst_o => gtFreeRst
      );

   -----------------------------
   --! GTY transceivers 
   -----------------------------
   u0_GtyCoreWrapper : entity work.GtyCoreWrapper
      generic map (
         TPD_G        => TPD_G,
         CLK_CORR_2_G => CLK_CORR_2_C,
         CLK_CORR_1_G => CLK_CORR_1_C, 
         N_CHAN_G     => 2*10+8, -- 2*10+2,
         HOLD_TIME_G  => RST_HOLD_TIME_C)
      port map (
         gtRst_i        => gtFreeRst,
         gtRefClk_i     => gtRefClk,
         gtFreeClk_i    => gtFreeClk,
         gtUsrClk_o     => clk,
         gtUsrRst_o     => rst,
         gtCommAligEn_i => '1',
         gtTxPol_i      => '0',
         gtRxPol_i      => '0',
         gtTxP_o        => gtTxP_o,
         gtTxN_o        => gtTxN_o,
         gtRxP_i        => gtRxP_i,
         gtRxN_i        => gtRxN_i,
         gtClkCorrPer_i => toSlv(CLK_CORR_PER_C,16),
         gtRx_o         => gtRx,
         gtTx_i         => gtTx,
         loopback_i     => (others => '0'));

   -----------------------------
   --! The main CMU logic 
   -----------------------------
   u0_CmuCore : entity work.CmuCore
      generic map (
         TPD_G        => TPD_G,
         BUILD_INFO_G => BUILD_INFO_G
      )
      port map (
         clk_i  => clk,
         rst_i  => rst,
         gtRx_i => gtRx,
         gtTx_o => gtTx
      );

end structure;