---------------------------------------------------------------------------------------------------
--! @brief     CMU Clock Reset module.
--!
--! @details   
--!   The MMCM is used to change the clock frequency of the free running clock (gtFreeClk) from 156.25 MHz to 100 MHz, 
--!   The module buffers 125 MHz GT reference clocks (gtRefClk).
--!
--!   The module also handles the power-up reset (gtFreeRst). The circuit is reset in the following cases:
--!   - at power-up the reset is held high and released after approximately 2 * HOLD_TIME_G * 6.4 ns + 1 * HOLD_TIME_G * 10 ns.
--!   - if the MMCM locked (locked) drops. The reset is released after approximately 1 * HOLD_TIME_G * 6.4 ns + 1 * HOLD_TIME_G * 10 ns.
--!   
--!   Why do we reset the MMCM if the locked drops?
--!   UG572 (v1.9) October 31, 2019 in table 3-3 page 48 the manual sais: "The MMCM must be reset after LOCKED is deasserted."
--!
--!
--! @author Uros Legat, Cosylab (uros.legat@cosylab.com)
--!
--! @date Sep 06 2019 created
--! @date Dec 20 2019 last modify
--!
--! @version v1.0
--!
--! @file CmuClkRst.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_arith.all;
library UNISIM;
use UNISIM.VComponents.all;
use work.CslStdRtlPkg.all;

---------------------------------------------------------------------------------------------------
entity CmuClkRst is
   generic (
      TPD_G       : time     := 1 ns;
      HOLD_TIME_G : positive := 1000 --! hold time for PowerupRst module
   );
   port (
      userClkP_i   : in sl; --! 156.25 MHz user clock P line
      userClkN_i   : in sl; --! 156.25 MHz user clock N line
      gtRefClkP_i  : in slv(2 downto 0); --! 125 MHz reference clock P line
      gtRefClkN_i  : in slv(2 downto 0); --! 125 MHz reference clock N line
      gtRefClk_o   : out slv(2 downto 0); --! 125 MHz Single ended reference clock
      
      gtFreeClk_o : out sl; --! 100 MNHz free running clock for GT transceivers
      gtFreeRst_o : out sl  --! reset for free running clock for GT transceivers
   );
end CmuClkRst;
---------------------------------------------------------------------------------------------------
architecture rtl of CmuClkRst is

   --! internal signals
   signal clkBuf     : sl;
   signal clkBufRst  : sl;
   
   signal locked        : sl;
   signal lockedSync    : sl;   
   signal lockedFed     : sl;
   signal clk           : sl;
   signal rst           : sl;

---------------------------------------------------------------------------------------------------
begin

   --! 125.00 GT Reference clocks from bank 227-1 and 229-1
   ------------------------------------------------------------
   REF_CLK_GEN : for i in 2 downto 0 generate
      u0_IBUFDS_GTE4 : IBUFDS_GTE4
         generic map (
            REFCLK_EN_TX_PATH  => '0',  -- Refer to Transceiver User Guide
            REFCLK_HROW_CK_SEL => "00", -- Refer to Transceiver User Guide
            REFCLK_ICNTL_RX    => "00"  -- Refer to Transceiver User Guide
         )
         port map (
            O     => gtRefClk_o(i),   -- 1-bit output: Refer to Transceiver User Guide
            ODIV2 => open,         -- The reference clock is not used internally
            CEB   => '0',          -- 1-bit input: Refer to Transceiver User Guide
            I     => gtRefClkP_i(i),  -- 1-bit input: Refer to Transceiver User Guide
            IB    => gtRefClkN_i(i)   -- 1-bit input: Refer to Transceiver User Guide
         );
   end generate REF_CLK_GEN;


   --! 156.25 LVDS user clock input
   ------------------------------------------------------------
   u0_IBUFDS : IBUFGDS
      port map (
         O     => clkBuf,   
         I     => userClkP_i,
         IB    => userClkN_i
      );
      
   --! Synchronise the locked back to userClk domain
   u0_CslSync : entity work.CslSync
      port map (
         clk_i => clkBuf,
         rst_i => '0',
         sig_i => locked,
         sig_o => lockedSync
      );
      
   --! Detect if locked drops.
   u0_CslEdgeDet : entity work.CslEdgeDet
      generic map (
         TPD_G          => TPD_G,
         FALLING_EDGE_G => true
      )
      port map (
         clk_i => clkBuf,
         rst_i => '0',
         sig_i => lockedSync,
         sig_o => lockedFed
      );

   --! Reset the MMCM if the locked drops.
   u0_PowerupRst : entity work.PowerupRst
      generic map (
         TPD_G         => TPD_G,
         INV_RST_POL_G => false,
         HOLD_TIME_G   => HOLD_TIME_G
      )
      port map (
         clk_i    => clkBuf,
         extRst_i => lockedFed,
         rst_o    => clkBufRst
      );

   --! MMCM component to generate 100 MHz clock 
   u0_ClkRstMMCM : entity work.ClkRstMMCM
      port map (
         reset => clkBufRst,
         clk_i => clkBuf,         
         clk_o => clk,  
         -- Status and control signals                
         locked => locked
      );

   --! Release the reset after the lock of the MMCM is established.
   u0_Sync : entity work.PowerupRst
      generic map (
         TPD_G         => TPD_G,
         INV_RST_POL_G => false,      --! external reset reverse polarity
         HOLD_TIME_G   => HOLD_TIME_G --! hold time for reset signal
      )
      port map (
         clk_i    => clk,         --! clock input signal
         extRst_i => not(locked), --! external reset signal
         rst_o    => rst          --! reset output after Hold time + sync pipe (3 c-c) + DFF
      );

   --! Free running clock 100MHz  
   gtFreeClk_o <= clk;
   gtFreeRst_o <= rst;   
   --------------------------------------
end rtl;
---------------------------------------------------------------------------------------------------