---------------------------------------------------------------------------------------------------
--! @brief     CMU Core Package 
--! @details    
--!   All CMU types, functions, declarations are in this file. 
--!    
--! @author Uros Legat, Cosylab (uros.legat@cosylab.com)
--!
--! @date Oct 07 2019 created
--! @date Oct 07 2019 last modify
--!
--! @version v1.0
--!
--! @file CmuCorePkg.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
use work.FiberPkg.all;
---------------------------------------------------------------------------------------------------
package CmuCorePkg is
   ------------------------------------------------------------   
   -- Constant definitions
   ------------------------------------------------------------

   --! Clock correction constants
   ------------------------------------------------------------
   constant CLK_CORR_2_C   : slv(8 downto 0) := '1' & x"FD";
   constant CLK_CORR_1_C   : slv(8 downto 0) := '1' & x"BC";
   constant CLK_CORR_PER_C : positive        := 2500;
   ------------------------------------------------------------       
   --! Reset hold time
   --! Time of one powerup reset module
   --! Three powerup reset modules are employed to generate "rst" signal
   constant RST_HOLD_TIME_C : positive := 200;
   ------------------------------------------------------------
   --  Time that the Cmu stays in Init state.
   --  INIT_TIME_C/2 Clear error counters.
   --  INIT_TIME_C/2 Release error counters. Self-test.
   constant INIT_TIME_C     : positive := 10E+8;
   constant SIM_INIT_TIME_C : positive := 1000;
   ------------------------------------------------------------
   --! Number of beamlines
   constant N_BL_C : positive := 6;
   ------------------------------------------------------------
   --! FEC command
   constant FEC_RESET_C : slv(0 downto 0) := "1";
   ------------------------------------------------------------
   --! FSM states
   constant INIT_C : slv(1 downto 0) := "00";
   constant OPER_C : slv(1 downto 0) := "01";
   constant FAIL_C : slv(1 downto 0) := "10";
   ------------------------------------------------------------
   --! Communication constants
   constant FIBER_LINK_TIME_C : positive := 100000;
   constant SFED_WD_TIME_C    : positive := 3000;
   constant ITU_PERIOD_C      : positive := 2500;
   constant ITM_PERIOD_C      : positive := 500;
   constant DDS_PERIOD_C      : positive := 1000;

   constant SIM_FIBER_LINK_TIME_C : positive := 250;
   constant SIM_SFED_WD_TIME_C    : positive := 300;
   constant SIM_ITU_PERIOD_C      : positive := 256;

   --! Axi-Lite slave response timeout
   constant AXIL_REG_TOUT_C : positive := 1000;

   ------------------------------------------------------------
   ------------------------------------------------------------

   ------------------------------------------------------------
   --! FEC stat type
   type FecStatType is record
      err            : sl;
      link           : sl;
      cfgCrcCnt      : slv(31 downto 0);
      cfgLenCnt      : slv(31 downto 0);
      cfgDropCnt     : slv(31 downto 0);
      gtDecErrCnt    : slv(31 downto 0);
      gtDispErrCnt   : slv(31 downto 0);
      gtByteReAlig   : slv(31 downto 0);
      gtCdrStableCnt : slv(31 downto 0);
   end record FecStatType;

   constant FEC_STAT_INIT_C : FecStatType := (
         err            => '0',
         link           => '0',
         cfgCrcCnt      => (others => '0'),
         cfgLenCnt      => (others => '0'),
         cfgDropCnt     => (others => '0'),
         gtDecErrCnt    => (others => '0'),
         gtDispErrCnt   => (others => '0'),
         gtByteReAlig   => (others => '0'),
         gtCdrStableCnt => (others => '0')
      );

   ------------------------------------------------------------

   ------------------------------------------------------------
   --! FEC Ctrl type
   type FecCtrlType is record
      clearStat  : sl;
      cmd        : slv(0 downto 0);
      sfedEnable : slv(5 downto 0);
   end record FecCtrlType;

   constant FEC_CTRL_INIT_C : FecCtrlType := (
         clearStat => '0',
         cmd       => (others => '0'),
         -- Note: only first five beamlines are enabled at startup
         sfedEnable => "011111"
      );

   ------------------------------------------------------------

   ------------------------------------------------------------
   --! Sfed stat type
   type SfedStatType is record
      err           : sl;
      link          : sl;
      crcCnt        : slv(31 downto 0);
      lenCnt        : slv(31 downto 0);
      dropCnt       : slv(31 downto 0);
      tmstpStallCnt : slv(31 downto 0);
      toutCnt       : slv(31 downto 0);
      ddsLinkCnt    : slv(31 downto 0);
      ddsParityCnt  : slv(31 downto 0);

      gtDecErrCnt    : slv(31 downto 0);
      gtDispErrCnt   : slv(31 downto 0);
      gtByteReAlig   : slv(31 downto 0);
      gtCdrStableCnt : slv(31 downto 0);
   end record SfedStatType;

   constant SFED_STAT_INIT_C : SfedStatType := (
         err           => '0',
         link          => '0',
         crcCnt        => (others => '0'),
         lenCnt        => (others => '0'),
         dropCnt       => (others => '0'),
         tmstpStallCnt => (others => '0'),
         toutCnt       => (others => '0'),
         ddsLinkCnt    => (others => '0'),
         ddsParityCnt  => (others => '0'),

         gtDecErrCnt    => (others => '0'),
         gtDispErrCnt   => (others => '0'),
         gtByteReAlig   => (others => '0'),
         gtCdrStableCnt => (others => '0')
      );

   type SfedStatArray is array (natural range<>) of SfedStatType;
   ------------------------------------------------------------
      ------------------------------------------------------------
   --! ITU message type
   -- type ItuMsgType is record
      -- strobe       : sl;
      -- statFlags    : slv(11 downto 0);
      -- setpAbsCurrX : Slv32Array(5 downto 0);
      -- measAbsCurrX : Slv32Array(5 downto 0);
      -- setpRelCurrX : Slv16Array(5 downto 0);
      -- measRelCurrX : Slv16Array(5 downto 0);
      -- setpAbsCurrY : Slv32Array(5 downto 0);
      -- measAbsCurrY : Slv32Array(5 downto 0);
      -- setpRelCurrY : Slv16Array(5 downto 0);
      -- measRelCurrY : Slv16Array(5 downto 0);
      -- timeStamp    : slv(63 downto 0);
   -- end record ItuMsgType;

   -- constant ITU_MSG_INIT_C : ItuMsgType := (
         -- strobe       => '0',
         -- statFlags    => (others => '0'),
         -- setpAbsCurrX => (others => (others => '0')),
         -- measAbsCurrX => (others => (others => '0')),
         -- setpRelCurrX => (others => (others => '0')),
         -- measRelCurrX => (others => (others => '0')),
         -- setpAbsCurrY => (others => (others => '0')),
         -- measAbsCurrY => (others => (others => '0')),
         -- setpRelCurrY => (others => (others => '0')),
         -- measRelCurrY => (others => (others => '0')),
         -- timeStamp    => (others => '0')
      -- );

   type ItuMsgType is record
      strobe       : sl;
    --  statFlagsX   : Slv3Array(5 downto 0);
    --  statFlagsY   : Slv3Array(5 downto 0);
    statFlagsX0			  : slv(2 downto 0);
    statFlagsX1			  : slv(2 downto 0);
	statFlagsX2			  : slv(2 downto 0);
	statFlagsX3			  : slv(2 downto 0);
	statFlagsX4			  : slv(2 downto 0);
	statFlagsX5			  : slv(2 downto 0);	
	statFlagsY0			  : slv(2 downto 0);
	statFlagsY1			  : slv(2 downto 0);
	statFlagsY2			  : slv(2 downto 0);
	statFlagsY3			  : slv(2 downto 0);
	statFlagsY4			  : slv(2 downto 0);
	statFlagsY5			  : slv(2 downto 0);  
    setpAbsCurrX : Slv32Array(5 downto 0);
    measAbsCurrX : Slv32Array(5 downto 0);	
    setpRelCurrX : Slv16Array(5 downto 0);
    measRelCurrX : Slv16Array(5 downto 0);
    timeStamp1X	 : Slv32Array(5 downto 0);
    timeStamp2X	 : Slv32Array(5 downto 0);
    timeStamp1Y	 : Slv32Array(5 downto 0);
    timeStamp2Y	 : Slv32Array(5 downto 0);    
--	timeStampX0	 : slv(63 downto 0);
--	timeStampX1	 : slv(63 downto 0);
--	timeStampX2	 : slv(63 downto 0);
--	timeStampX3	 : slv(63 downto 0);
--	timeStampX4	 : slv(63 downto 0);
--	timeStampX5	 : slv(63 downto 0);
    setpAbsCurrY : Slv32Array(5 downto 0);
    measAbsCurrY : Slv32Array(5 downto 0);
    setpRelCurrY : Slv16Array(5 downto 0);
    measRelCurrY : Slv16Array(5 downto 0);
--	timeStampY0	 : slv(63 downto 0);
--	timeStampY1	 : slv(63 downto 0);
--	timeStampY2	 : slv(63 downto 0);
--	timeStampY3	 : slv(63 downto 0);
--	timeStampY4	 : slv(63 downto 0);
--	timeStampY5	 : slv(63 downto 0);
    timeStamp    : slv(63 downto 0);
   end record ItuMsgType;

   constant ITU_MSG_INIT_C : ItuMsgType := (
         strobe       => '0',
--         statFlagsX    => (others => (others => '0')),
--         statFlagsY    => (others => (others => '0')),
         statFlagsX0  => (others => '0'),
         statFlagsX1  => (others => '0'),
         statFlagsX2  => (others => '0'),
         statFlagsX3  => (others => '0'),
         statFlagsX4  => (others => '0'),
         statFlagsX5  => (others => '0'),
         statFlagsY0  => (others => '0'),
         statFlagsY1  => (others => '0'),
         statFlagsY2  => (others => '0'),
         statFlagsY3  => (others => '0'),
         statFlagsY4  => (others => '0'),
         statFlagsY5  => (others => '0'),      
         setpAbsCurrX => (others => (others => '0')),
         measAbsCurrX => (others => (others => '0')),
         setpRelCurrX => (others => (others => '0')),
         measRelCurrX => (others => (others => '0')),
         timeStamp1X  => (others => (others => '0')),
         timeStamp2X  => (others => (others => '0')),
         timeStamp1Y  => (others => (others => '0')),
         timeStamp2Y  => (others => (others => '0')),
         
--		timeStampX0	 => (others => '0'),      
--		timeStampX1	 => (others => '0'),      
--		timeStampX2	 => (others => '0'),      
--		timeStampX3	 => (others => '0'),      
--		timeStampX4	 => (others => '0'),      
--		timeStampX5	 => (others => '0'),      
         setpAbsCurrY => (others => (others => '0')),		 
         measAbsCurrY => (others => (others => '0')),
         setpRelCurrY => (others => (others => '0')),
         measRelCurrY => (others => (others => '0')),
--		timeStampY0	 => (others => '0'),      
--		timeStampY1	 => (others => '0'),      
--		timeStampY2	 => (others => '0'),      
--		timeStampY3	 => (others => '0'),      
--		timeStampY4	 => (others => '0'),      
--		timeStampY5	 => (others => '0'),  
         timeStamp    => (others => '0')
      );


--  type ItuMsg1Type is record
--        statusItm0    : slv (2 downto 0);      
--        statusItm1    : slv (2 downto 0);        
--        statusItm2    : slv (2 downto 0);        
--        statusItm3    : slv (2 downto 0);        
--        statusItm4    : slv (2 downto 0);
--        statusItm5    : slv (2 downto 0);                        
--        dataItm0      : Slv32Array(9 downto 0);
--        dataItm1      : Slv32Array(9 downto 0);
--        dataItm2      : Slv32Array(9 downto 0);
--        dataItm3      : Slv32Array(9 downto 0);
--        dataItm4      : Slv32Array(9 downto 0);
--        dataItm5      : Slv32Array(9 downto 0);                  
--   end record ItuMsg1Type;
		
--   constant ITU_MSG1_INIT_C : ItuMsg1Type := (        
--         statusItm0     => (others => '0'),
--         statusItm1     => (others => '0'),
--         statusItm2     => (others => '0'),
--         statusItm3     => (others => '0'),
--         statusItm4     => (others => '0'),
--         statusItm5     => (others => '0'),         
--         dataItm0       => (others => (others => '0')),
--         dataItm1       => (others => (others => '0')),
--         dataItm2       => (others => (others => '0')),
--         dataItm3       => (others => (others => '0')),
--         dataItm4       => (others => (others => '0')),
--         dataItm5       => (others => (others => '0'))
         
--      );
	
	
   ------------------------------------------------------------
   
   --! Itm stat type
   type ItmStatType is record
      err           : sl;
      link          : sl;
      crcCnt        : slv(31 downto 0);
      lenCnt        : slv(31 downto 0);
      dropCnt       : slv(31 downto 0);
      tmstpStallCnt : slv(31 downto 0);
      toutCnt       : slv(31 downto 0);
      ddsLinkCnt    : slv(31 downto 0);
      ddsParityCnt  : slv(31 downto 0);

      gtDecErrCnt    : slv(31 downto 0);
      gtDispErrCnt   : slv(31 downto 0);
      gtByteReAlig   : slv(31 downto 0);
      gtCdrStableCnt : slv(31 downto 0);
   end record ItmStatType;

   constant ITM_STAT_INIT_C : ItmStatType := (
         err           => '0',
         link          => '0',
         crcCnt        => (others => '0'),
         lenCnt        => (others => '0'),
         dropCnt       => (others => '0'),
         tmstpStallCnt => (others => '0'),
         toutCnt       => (others => '0'),
         ddsLinkCnt    => (others => '0'),
         ddsParityCnt  => (others => '0'),

         gtDecErrCnt    => (others => '0'),
         gtDispErrCnt   => (others => '0'),
         gtByteReAlig   => (others => '0'),
         gtCdrStableCnt => (others => '0')
      );

   type ItmStatArray is array (natural range<>) of ItmStatType;
   ------------------------------------------------------------
   --! Dds stat type
   type DdsStatType is record
      err           : sl;
      link          : sl;
      crcCnt        : slv(31 downto 0);
      lenCnt        : slv(31 downto 0);
      dropCnt       : slv(31 downto 0);
      tmstpStallCnt : slv(31 downto 0);
      toutCnt       : slv(31 downto 0);
      ddsLinkCnt    : slv(31 downto 0);
      ddsParityCnt  : slv(31 downto 0);

      gtDecErrCnt    : slv(31 downto 0);
      gtDispErrCnt   : slv(31 downto 0);
      gtByteReAlig   : slv(31 downto 0);
      gtCdrStableCnt : slv(31 downto 0);
   end record DdsStatType;

   constant DDS_STAT_INIT_C : DdsStatType := (
         err           => '0',
         link          => '0',
         crcCnt        => (others => '0'),
         lenCnt        => (others => '0'),
         dropCnt       => (others => '0'),
         tmstpStallCnt => (others => '0'),
         toutCnt       => (others => '0'),
         ddsLinkCnt    => (others => '0'),
         ddsParityCnt  => (others => '0'),

         gtDecErrCnt    => (others => '0'),
         gtDispErrCnt   => (others => '0'),
         gtByteReAlig   => (others => '0'),
         gtCdrStableCnt => (others => '0')
      );

   type DdsStatArray is array (natural range<>) of DdsStatType;
   
   
   
   ------------------------------------------------------------
   --! FSM stat type
   type FsmStatType is record
      state : slv(1 downto 0);
   end record FsmStatType;

   constant FSM_STAT_INIT_C : FsmStatType := (
         state => (others => '0'));
 
   ------------------------------------------------------------

   ------------------------------------------------------------
   --! Sfed message type
   type SfedMsgType is record
      strobe      : sl;
      statFlags   : slv(5 downto 0);
      setpAbsCurr : slv(31 downto 0);
      measAbsCurr : slv(31 downto 0);
      setpRelCurr : slv(15 downto 0);
      measRelCurr : slv(15 downto 0);
      timeStamp   : slv(63 downto 0);
   end record SfedMsgType;

   constant SFED_MSG_INIT_C : SfedMsgType := (
         strobe      => '0',
         statFlags   => (others => '0'),
         setpAbsCurr => (others => '0'),
         measAbsCurr => (others => '0'),
         setpRelCurr => (others => '0'),
         measRelCurr => (others => '0'),
         timeStamp   => (others => '0')
      );

   type SfedMsgArray is array (natural range<>) of SfedMsgType;

   ------------------------------------------------------------   

   ------------------------------------------------------------
   --! CFG message type
   type CfgMsgType is record
      strobe : sl;
      tid    : slv(7 downto 0);
      cmd    : slv(7 downto 0);
      rsp    : sl;
      dev    : slv(31 downto 0);
      len    : slv(31 downto 0);
      addr   : slv(31 downto 0);
      acsErr : sl;
      cmdErr : sl;
      touErr : sl;
      devErr : sl;
      adrErr : sl;
      lenErr : sl;
      crcErr : sl;
      parErr : sl;
      data   : Slv32Array(15 downto 0);
   end record CfgMsgType;

   constant CFG_MSG_INIT_C : CfgMsgType := (
         strobe => '0',
         tid    => (others => '0'),
         cmd    => (others => '0'),
         rsp    => '0',
         dev    => (others => '0'),
         len    => (others => '0'),
         addr   => (others => '0'),
         acsErr => '0',
         cmdErr => '0',
         touErr => '0',
         devErr => '0',
         adrErr => '0',
         lenErr => '0',
         crcErr => '0',
         parErr => '0',
         data   => (others => (others => '0'))
      );

  ------------------------------------------------------------
   --! Itm message type
   type ItmMsgType is record
      strobe                  : sl;
      statFlags               : slv(15 downto 0);
      timeStamp               : slv(15 downto 0);
      countsLow               : slv(31 downto 0);
      countsHigh              : slv(31 downto 0);
      tempIn                  : slv(31 downto 0);
      tempOut                 : slv(31 downto 0);
      pressureIn              : slv(31 downto 0);
      pressureOut             : slv(31 downto 0);
      fluxIn                  : slv(31 downto 0);
      fluxOut                 : slv(31 downto 0);
      itmCrc                  : slv(31 downto 0);      
   end record ItmMsgType;

   constant ITM_MSG_INIT_C : ItmMsgType := (
      strobe                  =>  '0',
      statFlags               =>  (others => '0'),
      timeStamp               =>  (others => '0'),
      countsLow               =>  (others => '0'),
      countsHigh              =>  (others => '0'),
      tempIn                  =>  (others => '0'),
      tempOut                 =>  (others => '0'),
      pressureIn              =>  (others => '0'),
      pressureOut             =>  (others => '0'),
      fluxIn                  =>  (others => '0'),
      fluxOut                 =>  (others => '0'),
      itmCrc                  =>  (others => '0')
      );

   type ItmMsgArray is array (natural range<>) of ItmMsgType;



	type ItuMsg1Type is record           
	          strobe                  : sl;   
			  statFlagsItm0			  : slv(2 downto 0);
			  statFlagsItm1			  : slv(2 downto 0);
			  statFlagsItm2			  : slv(2 downto 0);
			  statFlagsItm3			  : slv(2 downto 0);
			  statFlagsItm4			  : slv(2 downto 0);
			  statFlagsItm5			  : slv(2 downto 0);			  
			  dataItm0                : ItmMsgType;
			  dataItm1                : ItmMsgType;
			  dataItm2                : ItmMsgType;
			  dataItm3                : ItmMsgType;
			  dataItm4                : ItmMsgType;
			  dataItm5                : ItmMsgType;			  
			  timeStamp               : slv(63 downto 0);			                     	  
		end record ItuMsg1Type;
		
       constant ITU_MSG1_INIT_C : ItuMsg1Type := (    
             strobe                  =>  '0',    
             statFlagsItm0     => (others => '0'),
             statFlagsItm1     => (others => '0'),
             statFlagsItm2     => (others => '0'),
             statFlagsItm3     => (others => '0'),
             statFlagsItm4     => (others => '0'),
             statFlagsItm5     => (others => '0'),         
             dataItm0          => ITM_MSG_INIT_C,
             dataItm1          => ITM_MSG_INIT_C,
             dataItm2          => ITM_MSG_INIT_C,
             dataItm3          => ITM_MSG_INIT_C,
             dataItm4          => ITM_MSG_INIT_C,
             dataItm5          => ITM_MSG_INIT_C,
             timeStamp         => (others => '0')             
          );
   ------------------------------------------------------------  
   
    ------------------------------------------------------------
   --! DDS message type
   type DdsMsgType is record
      strobe                  : sl;
      ddsInstId               : slv(3 downto 0);
      msgId               	  : slv(7 downto 0);
	  isDelActive             : sl;
	  isPosValid	          : sl;
      currLayerId             : slv(7 downto 0);
	  currSpotId              : slv(13 downto 0);	  
      posX                    : slv(12 downto 0);
	  posY                    : slv(12 downto 0);	  
      pad	                  : slv(1 downto 0);      
   end record DdsMsgType;

   constant DDS_MSG_INIT_C : DdsMsgType := (
      strobe                =>  '0',
      ddsInstId             =>  (others => '0'),
      msgId               	=>  (others => '0'),
      isDelActive           =>  '0',
      isPosValid            =>  '0',
      currLayerId           =>  (others => '0'),
      currSpotId            =>  (others => '0'),
      posX              	=>  (others => '0'),
      posY             		=>  (others => '0'),
      pad                  =>  (others => '0')      
      );

   type DdsMsgArray is array (natural range<>) of DdsMsgType;

	type ItuMsg2Type is record           
	          strobe                  : sl;   
			  statFlagsDds0			  : slv(2 downto 0);
			  statFlagsDds1			  : slv(2 downto 0);
			  statFlagsDds2			  : slv(2 downto 0);
			  statFlagsDds3			  : slv(2 downto 0);
			  statFlagsDds4			  : slv(2 downto 0);
			  statFlagsDds5			  : slv(2 downto 0);			  
			  dataDds0                : DdsMsgType;
			  dataDds1                : DdsMsgType;
			  dataDds2                : DdsMsgType;
			  dataDds3                : DdsMsgType;
			  dataDds4                : DdsMsgType;
			  dataDds5                : DdsMsgType;			  
			  timeStamp               : slv(63 downto 0);			                     	  
		end record ItuMsg2Type;
		
       constant ITU_MSG2_INIT_C : ItuMsg2Type := (    
             strobe                  =>  '0',    
             statFlagsDds0     => (others => '0'),
             statFlagsDds1     => (others => '0'),
             statFlagsDds2     => (others => '0'),
             statFlagsDds3     => (others => '0'),
             statFlagsDds4     => (others => '0'),
             statFlagsDds5     => (others => '0'),         
             dataDds0          => DDS_MSG_INIT_C,
             dataDds1          => DDS_MSG_INIT_C,
             dataDds2          => DDS_MSG_INIT_C,
             dataDds3          => DDS_MSG_INIT_C,
             dataDds4          => DDS_MSG_INIT_C,
             dataDds5          => DDS_MSG_INIT_C,
             timeStamp         => (others => '0')             
          );
   ------------------------------------------------------------  
   -------------------------------------------------------------   
   --! Functions section 
   -------------------------------------------------------------
   --! CMU message function
   function encSfedMsg (msg : Slv32Array(7 downto 0); strobe : sl) return SfedMsgType;
   function encItmMsg (msg : Slv32Array(9 downto 0); strobe : sl) return ItmMsgType;
   function encDdsMsg (msg : Slv32Array(1 downto 0); strobe : sl) return DdsMsgType;
   ------------------------------------------------------------
   --! CFG message functions
   function encCfgFec (msg    : Slv32Array(23 downto 0); strobe : sl) return CfgMsgType;
   function decCfgFec (CfgMsg : CfgMsgType) return Slv32Array;
   ------------------------------------------------------------
   --! ITU message functions
   function encHpdTxFec (ituMsg : ItuMsgType) return Slv32Array;
   function encHpdTxItu (ituMsg : ItuMsgType) return Slv32Array;
   function encItuMsg (msg      : Slv32Array(39 downto 0); strobe : sl) return ItuMsgType;
   function encFecMsg (msg      : Slv32Array(44 downto 0); strobe : sl) return ItuMsgType;
  ------------------------------------------------------------
   --! ITM message function
   function encHpdTxItmItu (ituMsg : ItuMsg1Type) return Slv32Array;
   function encItmItuMsg (msg  : Slv32Array(62 downto 0); strobe : sl) return ItuMsg1Type;
   
   --! DDS message function
   function encHpdTxDdsItu (ituMsg : ItuMsg2Type) return Slv32Array;
   function encDdsItuMsg (msg  : Slv32Array(14 downto 0); strobe : sl) return ItuMsg2Type;
   
   
------------------------------------------------------------
   
end package CmuCorePkg;
---------------------------------------------------------------------------------------------------
package body CmuCorePkg is

   ------------------------------------------------------------
   --! CFG message functions
   function decCfgFec (CfgMsg : CfgMsgType) return Slv32Array is
      variable msg : Slv32Array(23 downto 0);
   begin
      msg(0)(31 downto 24) := cfgMsg.tid;
      msg(0)(23 downto 8)  := (others => '0');
      msg(0)(7 downto 0)   := cfgMsg.cmd;
      msg(1)(31 downto 0)  := cfgMsg.dev;
      msg(2)(31 downto 0)  := cfgMsg.addr;
      msg(3)(31 downto 0)  := cfgMsg.len;
      msg(4)(0)            := cfgMsg.acsErr;
      msg(4)(1)            := cfgMsg.cmdErr;
      msg(4)(2)            := cfgMsg.touErr;
      msg(4)(3)            := cfgMsg.devErr;
      msg(4)(4)            := cfgMsg.adrErr;
      msg(4)(5)            := cfgMsg.lenErr;
      msg(4)(6)            := cfgMsg.crcErr;
      msg(4)(7)            := cfgMsg.parErr;
      msg(4)(23 downto 8)  := (others => '0');
      msg(4)(31 downto 24) := "0000000" & cfgMsg.rsp;
      msg(5)               := (others => '0');
      msg(6)               := (others => '0');
      msg(7)               := (others => '0');
      for i in 0 to 15 loop
         msg(8+i) := CfgMsg.data(i);
      end loop ;
      return msg ;
   end function decCfgFec;

   function encCfgFec (msg : Slv32Array(23 downto 0); strobe : sl) return CfgMsgType is
      variable CfgMsg : CfgMsgType;
   begin
      cfgMsg.strobe := strobe;
      cfgMsg.tid    := msg(0)(31 downto 24);
      cfgMsg.cmd    := msg(0)(7 downto 0);
      cfgMsg.rsp    := '0';
      cfgMsg.dev    := msg(1)(31 downto 0);
      cfgMsg.addr   := msg(2)(31 downto 0);
      cfgMsg.len    := msg(3)(31 downto 0);
      cfgMsg.acsErr := '0';
      cfgMsg.cmdErr := '0';
      cfgMsg.touErr := '0';
      cfgMsg.devErr := '0';
      cfgMsg.adrErr := '0';
      cfgMsg.lenErr := '0';
      cfgMsg.crcErr := '0';
      cfgMsg.parErr := '0';
      for i in 0 to 15 loop
         CfgMsg.data(i) := msg(8+i);
      end loop ;
      return CfgMsg ;
   end function encCfgFec;

   ------------------------------------------------------------

   ------------------------------------------------------------
   --! SFED message functions
   function encSfedMsg (msg : Slv32Array(7 downto 0); strobe : sl) return SfedMsgType is
      variable sfedMsg : SfedMsgType;
   begin
      sfedMsg.strobe                  := strobe;
      sfedMsg.statFlags               := msg(0)(5 downto 0);
      sfedMsg.setpAbsCurr             := msg(1)(31 downto 0);
      sfedMsg.measAbsCurr             := msg(2)(31 downto 0);
      sfedMsg.setpRelCurr             := msg(3)(15 downto 0);
      sfedMsg.measRelCurr             := msg(4)(15 downto 0);
      sfedMsg.timeStamp(31 downto 0)  := msg(6);
      sfedMsg.timeStamp(63 downto 32) := msg(7);
      return SfedMsg ;
   end function encSfedMsg;

   ------------------------------------------------------------
   --! ITU message functions
--   function encItuMsg (msg : Slv32Array(20 downto 0); strobe : sl) return ItuMsgType is
--      variable ItuMsg : ItuMsgType;
--   begin
--      ItuMsg.strobe    := strobe;
--      ItuMsg.statFlags := msg(0)(11 downto 0);
--      for i in 0 to 5 loop
--         ItuMsg.measRelCurrX(i) := msg(1 + i*2)(31 downto 16);
--         ItuMsg.setpRelCurrX(i) := msg(1 + i*2)(15 downto 0);
--      end loop;
--      for i in 0 to 5 loop
--         ItuMsg.measRelCurrY(i) := msg(2 + i*2)(31 downto 16);
--         ItuMsg.setpRelCurrY(i) := msg(2 + i*2)(15 downto 0);
--      end loop;
--      ItuMsg.timeStamp(31 downto 0)  := msg(19);
--      ItuMsg.timeStamp(63 downto 32) := msg(20);
--      return ItuMsg ;
--   end function encItuMsg;

   function encItuMsg (msg : Slv32Array(39 downto 0); strobe : sl) return ItuMsgType is
      variable ItuMsg : ItuMsgType;
   begin
      ItuMsg.strobe    := strobe;
--      for i in 0 to 4 loop
--      ItuMsg.statFlagsX(i) := msg(0)(11 downto 0);
--      ItuMsg.statFlagsY(i) := msg(0)(11 downto 0);            
--      end loop;
      ItuMsg.statFlagsX0 := msg(0)(2 downto 0);
      ItuMsg.statFlagsY0 := msg(0)(5 downto 3);
      ItuMsg.statFlagsX1 := msg(0)(8 downto 6);
      ItuMsg.statFlagsY1 := msg(0)(11 downto 9);
      ItuMsg.statFlagsX2 := msg(0)(14 downto 12);
      ItuMsg.statFlagsY2 := msg(0)(17 downto 15);
      ItuMsg.statFlagsX3 := msg(0)(20 downto 18);
      ItuMsg.statFlagsY3 := msg(0)(23 downto 21);
      ItuMsg.statFlagsX4 := msg(0)(26 downto 24);
      ItuMsg.statFlagsY4 := msg(0)(29 downto 27);
      ItuMsg.statFlagsX5 := msg(1)(0) & msg(0)(31 downto 30);
      ItuMsg.statFlagsY5 := msg(1)(3 downto 1);      
      
      for i in 0 to 5 loop
         ItuMsg.measRelCurrX(i) := msg(2 + i*6)(31 downto 16);
         ItuMsg.setpRelCurrX(i) := msg(2 + i*6)(15 downto 0);
      end loop;
      
      for i in 0 to 5 loop
         ItuMsg.timeStamp1X(i) := msg(3 + i*6);
         ItuMsg.timeStamp2X(i) := msg(4 + i*6);
      end loop;
      
      for i in 0 to 5 loop
         ItuMsg.measRelCurrY(i) := msg(5 + i*6)(31 downto 16);
         ItuMsg.setpRelCurrY(i) := msg(5 + i*6)(15 downto 0);
      end loop;
      
      for i in 0 to 5 loop
         ItuMsg.timeStamp1Y(i) := msg(6 + i*6);
         ItuMsg.timeStamp2Y(i) := msg(7 + i*6);
      end loop;
      
      ItuMsg.timeStamp(31 downto 0)  := msg(38);
      ItuMsg.timeStamp(63 downto 32) := msg(39);
      return ItuMsg ;
   end function encItuMsg;
   
--   function encFecMsg (msg : Slv32Array(44 downto 0); strobe : sl) return ItuMsgType is
--      variable FecMsg : ItuMsgType;
--   begin
--      FecMsg.strobe    := strobe;
--      FecMsg.statFlags := msg(0)(11 downto 0);
--      for i in 0 to 5 loop
--         FecMsg.measRelCurrX(i) := msg(1 + i*6)(31 downto 16);
--         FecMsg.setpRelCurrX(i) := msg(1 + i*6)(15 downto 0);
--         FecMsg.measAbsCurrX(i) := msg(2 + i*6)(31 downto 0);
--         FecMsg.setpAbsCurrX(i) := msg(3 + i*6)(31 downto 0);
--         FecMsg.measRelCurrY(i) := msg(4 + i*6)(31 downto 16);
--         FecMsg.setpRelCurrY(i) := msg(4 + i*6)(15 downto 0);
--         FecMsg.measAbsCurrY(i) := msg(5 + i*6)(31 downto 0);
--         FecMsg.setpAbsCurrY(i) := msg(6 + i*6)(31 downto 0);



--      end loop;
--      FecMsg.timeStamp(31 downto 0)  := msg(43);
--      FecMsg.timeStamp(63 downto 32) := msg(44);
--      return FecMsg ;
--   end function encFecMsg;

 function encFecMsg (msg : Slv32Array(44 downto 0); strobe : sl) return ItuMsgType is
      variable FecMsg : ItuMsgType;
   begin
      FecMsg.strobe    := strobe;
     -- FecMsg.statFlags := msg(0)(11 downto 0);
     
      for i in 0 to 5 loop
         FecMsg.measRelCurrX(i) := msg(1 + i*6)(31 downto 16);
         FecMsg.setpRelCurrX(i) := msg(1 + i*6)(15 downto 0);
         FecMsg.measAbsCurrX(i) := msg(2 + i*6)(31 downto 0);
         FecMsg.setpAbsCurrX(i) := msg(3 + i*6)(31 downto 0);
         FecMsg.measRelCurrY(i) := msg(4 + i*6)(31 downto 16);
         FecMsg.setpRelCurrY(i) := msg(4 + i*6)(15 downto 0);
         FecMsg.measAbsCurrY(i) := msg(5 + i*6)(31 downto 0);
         FecMsg.setpAbsCurrY(i) := msg(6 + i*6)(31 downto 0);



      end loop;
      FecMsg.timeStamp(31 downto 0)  := msg(43);
      FecMsg.timeStamp(63 downto 32) := msg(44);
      return FecMsg ;
   end function encFecMsg;


   function encHpdTxFec (ituMsg : ItuMsgType) return Slv32Array is
      variable hpdTxFec : Slv32Array(44 downto 0);
   begin
      -- Status
      hpdTxFec(0)(31 downto 12) := (others => '0');
--      hpdTxFec(0)(11 downto 0)  := ituMsg.statFlags;
      -- Relative currents 
      for i in 0 to 5 loop
         hpdTxFec(1 + i*6)(31 downto 16) := ituMsg.measRelCurrX(i);
         hpdTxFec(1 + i*6)(15 downto 0)  := ituMsg.setpRelCurrX(i);
         hpdTxFec(2 + i*6)(31 downto 0)  := ituMsg.measAbsCurrX(i);
         hpdTxFec(3 + i*6)(31 downto 0)  := ituMsg.setpAbsCurrX(i);
         hpdTxFec(4 + i*6)(31 downto 16) := ituMsg.measRelCurrY(i);
         hpdTxFec(4 + i*6)(15 downto 0)  := ituMsg.setpRelCurrY(i);
         hpdTxFec(5 + i*6)(31 downto 0)  := ituMsg.measAbsCurrY(i);
         hpdTxFec(6 + i*6)(31 downto 0)  := ituMsg.setpAbsCurrY(i);
      end loop ;
      -- Reserved for DCCT
      for i in 37 to 42 loop
         hpdTxFec(i) := (others => '0');
      end loop ;
      -- Timestamp
      hpdTxFec(43) := ituMsg.timeStamp(31 downto 0);
      hpdTxFec(44) := ituMsg.timeStamp(63 downto 32);
      return hpdTxFec ;
   end function encHpdTxFec;

   function encHpdTxItu (ituMsg : ItuMsgType) return Slv32Array is
      variable hpdTxItu : Slv32Array(39 downto 0);
   begin
      -- Status
      hpdTxItu(0)(2 downto 0)   := ItuMsg.statFlagsX0;      
      hpdTxItu(0)(5 downto 3)   := ItuMsg.statFlagsY0;
      hpdTxItu(0)(8 downto 6)   := ItuMsg.statFlagsX1;
      hpdTxItu(0)(11 downto 9)  := ItuMsg.statFlagsY1;
      hpdTxItu(0)(14 downto 12) := ItuMsg.statFlagsX2;
      hpdTxItu(0)(17 downto 15) := ItuMsg.statFlagsY2;
      hpdTxItu(0)(20 downto 18) := ItuMsg.statFlagsX3;
      hpdTxItu(0)(23 downto 21) := ItuMsg.statFlagsY3;
      hpdTxItu(0)(26 downto 24) := ItuMsg.statFlagsX4;
      hpdTxItu(0)(29 downto 27) := ItuMsg.statFlagsY4;
      hpdTxItu(0)(31 downto 30) := ItuMsg.statFlagsX5(1 downto 0) ;
      
      hpdTxItu(1)(0)            := ItuMsg.statFlagsX5(2);
      hpdTxItu(1)(3 downto 1)   := ItuMsg.statFlagsY5;      
      hpdTxItu(1)(31 downto 4)  := (others => '0');      
      
   --   hpdTxItu(0)(11 downto 0)  := ituMsg.statFlags;
   
      -- Relative currents 
      for i in 0 to 5 loop
         hpdTxItu(2 + i*6)(31 downto 16) := ituMsg.measRelCurrX(i);
         hpdTxItu(2 + i*6)(15 downto 0)  := ituMsg.setpRelCurrX(i);
      end loop;
	  
	  -- Time Stamps
      for i in 0 to 5 loop
         hpdTxItu(3 + i*6) := ituMsg.timeStamp1X(i);
         hpdTxItu(4 + i*6) := ituMsg.timeStamp2X(i);
      end loop;
	  
      for i in 0 to 5 loop
         hpdTxItu(5 + i*6)(31 downto 16) := ituMsg.measRelCurrY(i);
         hpdTxItu(5 + i*6)(15 downto 0)  := ituMsg.setpRelCurrY(i);
      end loop ;
	  
	  -- Time stamps
      for i in 0 to 5 loop
         hpdTxItu(6 + i*6) :=  ituMsg.timeStamp1Y(i);
         hpdTxItu(7 + i*6) :=  ituMsg.timeStamp2Y(i);
      end loop;
	  
--      -- Reserved for DCCT
--      for i in 14 to 19 loop
--         hpdTxItu(i) := (others => '0');
--      end loop ;
      -- Timestamp
      hpdTxItu(38) := ituMsg.timeStamp(31 downto 0);
      hpdTxItu(39) := ituMsg.timeStamp(63 downto 32);
      return hpdTxItu ;
   end function encHpdTxItu;


   ------------------------------------------------------------
   --! ITM message functions
   function encItmMsg (msg : Slv32Array(9 downto 0); strobe : sl) return ItmMsgType is
      variable itmMsg : ItmMsgType;
   begin
      itmMsg.strobe                  := strobe;
      itmMsg.statFlags               := msg(0)(15 downto 0);
      itmMsg.timeStamp               := msg(0)(31 downto 16);
      itmMsg.countsLow               := msg(1)(31 downto 0);
      itmMsg.countsHigh              := msg(2)(31 downto 0);
      itmMsg.tempIn                  := msg(3)(31 downto 0);
      itmMsg.tempOut                 := msg(4)(31 downto 0);
      itmMsg.pressureIn              := msg(5)(31 downto 0);
      itmMsg.pressureOut             := msg(6)(31 downto 0);
      itmMsg.fluxIn                  := msg(7)(31 downto 0);
      itmMsg.fluxOut                 := msg(8)(31 downto 0);
      itmMsg.itmCrc                  := msg(9)(31 downto 0);   
     return ItmMsg ;
   end function encItmMsg;

   ------------------------------------------------------------
  ------------------------------------------------------------
   --! ITU ITM  message functions
   function encItmItuMsg (msg : Slv32Array(62 downto 0); strobe : sl) return ItuMsg1Type is
      variable ItuMsg : ItuMsg1Type;
   begin
      ItuMsg.strobe         := strobe;     
      ituMsg.statFlagsItm5 := msg(0)(17 downto 15);
      ituMsg.statFlagsItm4 := msg(0)(14 downto 12);
      ituMsg.statFlagsItm3 := msg(0)(11 downto 9);
      ituMsg.statFlagsItm2 := msg(0)(8 downto 6);
      ituMsg.statFlagsItm1 := msg(0)(5 downto 3);
      ituMsg.statFlagsItm0 := msg(0)(2 downto 0);      
      
      ituMsg.dataItm0.timeStamp  := msg(1)(31 downto 16);
      ituMsg.dataItm0.statFlags  := msg(1)(15 downto 0); 
      ituMsg.dataItm0.countsLow  := msg(2)(31 downto 0);      
      ituMsg.dataItm0.countsHigh := msg(3)(31 downto 0);         
      ituMsg.dataItm0.tempIn     := msg(4)(31 downto 0);      
      ituMsg.dataItm0.tempOut    := msg(5)(31 downto 0);            
      ituMsg.dataItm0.pressureIn := msg(6)(31 downto 0);      
      ituMsg.dataItm0.pressureOut:= msg(7)(31 downto 0);      
      ituMsg.dataItm0.fluxIn     := msg(8)(31 downto 0);
      ituMsg.dataItm0.fluxOut    := msg(9)(31 downto 0);       
      ituMsg.dataItm0.itmCrc     := msg(10)(31 downto 0);   
      
      ituMsg.dataItm1.timeStamp  := msg(11)(31 downto 16);
      ituMsg.dataItm1.statFlags  := msg(11)(15 downto 0); 
      ituMsg.dataItm1.countsLow  := msg(12)(31 downto 0);      
      ituMsg.dataItm1.countsHigh := msg(13)(31 downto 0);         
      ituMsg.dataItm1.tempIn     := msg(14)(31 downto 0);      
      ituMsg.dataItm1.tempOut    := msg(15)(31 downto 0);            
      ituMsg.dataItm1.pressureIn := msg(16)(31 downto 0);      
      ituMsg.dataItm1.pressureOut:= msg(17)(31 downto 0);      
      ituMsg.dataItm1.fluxIn     := msg(18)(31 downto 0);
      ituMsg.dataItm1.fluxOut    := msg(19)(31 downto 0);       
      ituMsg.dataItm1.itmCrc     := msg(20)(31 downto 0);   
      
      ituMsg.dataItm2.timeStamp  := msg(21)(31 downto 16);
      ituMsg.dataItm2.statFlags  := msg(21)(15 downto 0); 
      ituMsg.dataItm2.countsLow  := msg(22)(31 downto 0);      
      ituMsg.dataItm2.countsHigh := msg(23)(31 downto 0);         
      ituMsg.dataItm2.tempIn     := msg(24)(31 downto 0);      
      ituMsg.dataItm2.tempOut    := msg(25)(31 downto 0);            
      ituMsg.dataItm2.pressureIn := msg(26)(31 downto 0);      
      ituMsg.dataItm2.pressureOut:= msg(27)(31 downto 0);      
      ituMsg.dataItm2.fluxIn     := msg(28)(31 downto 0);
      ituMsg.dataItm2.fluxOut    := msg(29)(31 downto 0);       
      ituMsg.dataItm2.itmCrc     := msg(30)(31 downto 0);   
      
      ituMsg.dataItm3.timeStamp  := msg(31)(31 downto 16);
      ituMsg.dataItm3.statFlags  := msg(31)(15 downto 0); 
      ituMsg.dataItm3.countsLow  := msg(32)(31 downto 0);      
      ituMsg.dataItm3.countsHigh := msg(33)(31 downto 0);         
      ituMsg.dataItm3.tempIn     := msg(34)(31 downto 0);      
      ituMsg.dataItm3.tempOut    := msg(35)(31 downto 0);            
      ituMsg.dataItm3.pressureIn := msg(36)(31 downto 0);      
      ituMsg.dataItm3.pressureOut:= msg(37)(31 downto 0);      
      ituMsg.dataItm3.fluxIn     := msg(38)(31 downto 0);
      ituMsg.dataItm3.fluxOut    := msg(39)(31 downto 0);       
      ituMsg.dataItm3.itmCrc     := msg(40)(31 downto 0);   
      
      ituMsg.dataItm4.timeStamp  := msg(41)(31 downto 16);
      ituMsg.dataItm4.statFlags  := msg(41)(15 downto 0); 
      ituMsg.dataItm4.countsLow  := msg(42)(31 downto 0);      
      ituMsg.dataItm4.countsHigh := msg(43)(31 downto 0);         
      ituMsg.dataItm4.tempIn     := msg(44)(31 downto 0);      
      ituMsg.dataItm4.tempOut    := msg(45)(31 downto 0);            
      ituMsg.dataItm4.pressureIn := msg(46)(31 downto 0);      
      ituMsg.dataItm4.pressureOut:= msg(47)(31 downto 0);      
      ituMsg.dataItm4.fluxIn     := msg(48)(31 downto 0);
      ituMsg.dataItm4.fluxOut    := msg(49)(31 downto 0);       
      ituMsg.dataItm4.itmCrc     := msg(50)(31 downto 0);   
      
      ituMsg.dataItm5.timeStamp  := msg(51)(31 downto 16);
      ituMsg.dataItm5.statFlags  := msg(51)(15 downto 0); 
      ituMsg.dataItm5.countsLow  := msg(52)(31 downto 0);      
      ituMsg.dataItm5.countsHigh := msg(53)(31 downto 0);         
      ituMsg.dataItm5.tempIn     := msg(54)(31 downto 0);      
      ituMsg.dataItm5.tempOut    := msg(55)(31 downto 0);            
      ituMsg.dataItm5.pressureIn := msg(56)(31 downto 0);      
      ituMsg.dataItm5.pressureOut:= msg(57)(31 downto 0);      
      ituMsg.dataItm5.fluxIn     := msg(58)(31 downto 0);
      ituMsg.dataItm5.fluxOut    := msg(59)(31 downto 0);       
      ituMsg.dataItm5.itmCrc     := msg(60)(31 downto 0);     
      
      ItuMsg.timeStamp(31 downto 0)  := msg(61);
      ItuMsg.timeStamp(63 downto 32) := msg(62);
      return ItuMsg ;
	 end function encItmItuMsg;
	  --------------------------------------------------------------------------------
   function encHpdTxItmItu (ituMsg : ItuMsg1Type) return Slv32Array is
      variable hpdTxItmItu : Slv32Array(62 downto 0);
   begin
      -- Status
      --1 word
      hpdTxItmItu(0)(31 downto 18):= (others => '0');
      hpdTxItmItu(0)(17 downto 15):= ituMsg.statFlagsItm5;
      hpdTxItmItu(0)(14 downto 12):= ituMsg.statFlagsItm4;
      hpdTxItmItu(0)(11 downto 9) := ituMsg.statFlagsItm3;
      hpdTxItmItu(0)(8 downto 6)  := ituMsg.statFlagsItm2;
      hpdTxItmItu(0)(5 downto 3)  := ituMsg.statFlagsItm1;
      hpdTxItmItu(0)(2 downto 0)  := ituMsg.statFlagsItm0;
      --2 word  - 11 word
      hpdTxItmItu(1)(31 downto 16):= ituMsg.dataItm0.timeStamp;
      hpdTxItmItu(1)(15 downto 0) := ituMsg.dataItm0.statFlags; 
      hpdTxItmItu(2)(31 downto 0) := ituMsg.dataItm0.countsLow;      
      hpdTxItmItu(3)(31 downto 0) := ituMsg.dataItm0.countsHigh;   
      hpdTxItmItu(4)(31 downto 0) := ituMsg.dataItm0.tempIn;      
      hpdTxItmItu(5)(31 downto 0) := ituMsg.dataItm0.tempOut;      
      hpdTxItmItu(6)(31 downto 0) := ituMsg.dataItm0.pressureIn;      
      hpdTxItmItu(7)(31 downto 0) := ituMsg.dataItm0.pressureOut;      
      hpdTxItmItu(8)(31 downto 0) := ituMsg.dataItm0.fluxIn;
      hpdTxItmItu(9)(31 downto 0) := ituMsg.dataItm0.fluxOut;       
      hpdTxItmItu(10)(31 downto 0):= ituMsg.dataItm0.itmCrc;      
      --12 word  - 21 word
      hpdTxItmItu(11)(31 downto 16):= ituMsg.dataItm1.timeStamp;
      hpdTxItmItu(11)(15 downto 0) := ituMsg.dataItm1.statFlags; 
      hpdTxItmItu(12)(31 downto 0) := ituMsg.dataItm1.countsLow;      
      hpdTxItmItu(13)(31 downto 0) := ituMsg.dataItm1.countsHigh;   
      hpdTxItmItu(14)(31 downto 0) := ituMsg.dataItm1.tempIn;      
      hpdTxItmItu(15)(31 downto 0) := ituMsg.dataItm1.tempOut;      
      hpdTxItmItu(16)(31 downto 0) := ituMsg.dataItm1.pressureIn;      
      hpdTxItmItu(17)(31 downto 0) := ituMsg.dataItm1.pressureOut;      
      hpdTxItmItu(18)(31 downto 0) := ituMsg.dataItm1.fluxIn;
      hpdTxItmItu(19)(31 downto 0) := ituMsg.dataItm1.fluxOut;       
      hpdTxItmItu(20)(31 downto 0) := ituMsg.dataItm1.itmCrc;
      --22 word  - 31 word
      hpdTxItmItu(21)(31 downto 16):= ituMsg.dataItm2.timeStamp;
      hpdTxItmItu(21)(15 downto 0) := ituMsg.dataItm2.statFlags; 
      hpdTxItmItu(22)(31 downto 0) := ituMsg.dataItm2.countsLow;      
      hpdTxItmItu(23)(31 downto 0) := ituMsg.dataItm2.countsHigh;   
      hpdTxItmItu(24)(31 downto 0) := ituMsg.dataItm2.tempIn;      
      hpdTxItmItu(25)(31 downto 0) := ituMsg.dataItm2.tempOut;      
      hpdTxItmItu(26)(31 downto 0) := ituMsg.dataItm2.pressureIn;      
      hpdTxItmItu(27)(31 downto 0) := ituMsg.dataItm2.pressureOut;      
      hpdTxItmItu(28)(31 downto 0) := ituMsg.dataItm2.fluxIn;
      hpdTxItmItu(29)(31 downto 0) := ituMsg.dataItm2.fluxOut;       
      hpdTxItmItu(30)(31 downto 0) := ituMsg.dataItm2.itmCrc;
       --32 word  - 41 word
      hpdTxItmItu(31)(31 downto 16):= ituMsg.dataItm3.timeStamp;
      hpdTxItmItu(31)(15 downto 0) := ituMsg.dataItm3.statFlags; 
      hpdTxItmItu(32)(31 downto 0) := ituMsg.dataItm3.countsLow;      
      hpdTxItmItu(33)(31 downto 0) := ituMsg.dataItm3.countsHigh;   
      hpdTxItmItu(34)(31 downto 0) := ituMsg.dataItm3.tempIn;      
      hpdTxItmItu(35)(31 downto 0) := ituMsg.dataItm3.tempOut;      
      hpdTxItmItu(36)(31 downto 0) := ituMsg.dataItm3.pressureIn;      
      hpdTxItmItu(37)(31 downto 0) := ituMsg.dataItm3.pressureOut;      
      hpdTxItmItu(38)(31 downto 0) := ituMsg.dataItm3.fluxIn;
      hpdTxItmItu(39)(31 downto 0) := ituMsg.dataItm3.fluxOut;       
      hpdTxItmItu(40)(31 downto 0):= ituMsg.dataItm3.itmCrc;      
      --42 word  - 51 word
      hpdTxItmItu(41)(31 downto 16):= ituMsg.dataItm4.timeStamp;
      hpdTxItmItu(41)(15 downto 0) := ituMsg.dataItm4.statFlags; 
      hpdTxItmItu(42)(31 downto 0) := ituMsg.dataItm4.countsLow;      
      hpdTxItmItu(43)(31 downto 0) := ituMsg.dataItm4.countsHigh;   
      hpdTxItmItu(44)(31 downto 0) := ituMsg.dataItm4.tempIn;      
      hpdTxItmItu(45)(31 downto 0) := ituMsg.dataItm4.tempOut;      
      hpdTxItmItu(46)(31 downto 0) := ituMsg.dataItm4.pressureIn;      
      hpdTxItmItu(47)(31 downto 0) := ituMsg.dataItm4.pressureOut;      
      hpdTxItmItu(48)(31 downto 0) := ituMsg.dataItm4.fluxIn;
      hpdTxItmItu(49)(31 downto 0) := ituMsg.dataItm4.fluxOut;       
      hpdTxItmItu(50)(31 downto 0) := ituMsg.dataItm4.itmCrc;
      --52 word  - 61 word
      hpdTxItmItu(51)(31 downto 16):= ituMsg.dataItm5.timeStamp;
      hpdTxItmItu(51)(15 downto 0) := ituMsg.dataItm5.statFlags; 
      hpdTxItmItu(52)(31 downto 0) := ituMsg.dataItm5.countsLow;      
      hpdTxItmItu(53)(31 downto 0) := ituMsg.dataItm5.countsHigh;   
      hpdTxItmItu(54)(31 downto 0) := ituMsg.dataItm5.tempIn;      
      hpdTxItmItu(55)(31 downto 0) := ituMsg.dataItm5.tempOut;      
      hpdTxItmItu(56)(31 downto 0) := ituMsg.dataItm5.pressureIn;      
      hpdTxItmItu(57)(31 downto 0) := ituMsg.dataItm5.pressureOut;      
      hpdTxItmItu(58)(31 downto 0) := ituMsg.dataItm5.fluxIn;
      hpdTxItmItu(59)(31 downto 0) := ituMsg.dataItm5.fluxOut;       
      hpdTxItmItu(60)(31 downto 0) := ituMsg.dataItm5.itmCrc;                                                                      
      -- Timestamp
      hpdTxItmItu(61)(31 downto 0) := ituMsg.timeStamp(31 downto 0);
      hpdTxItmItu(62)(31 downto 0) := ituMsg.timeStamp(63 downto 32);
      return hpdTxItmItu ;
   end function encHpdTxItmItu; 
 
    ------------------------------------------------------------
   --! DDS message functions   
   function encDdsMsg (msg : Slv32Array(1 downto 0); strobe : sl) return DdsMsgType is
      variable ddsMsg : DdsMsgType;
   begin
      ddsMsg.strobe             := strobe;
      ddsMsg.ddsInstId          := msg(0)(3 downto 0);
      ddsMsg.msgId              := msg(0)(11 downto 4);
      ddsMsg.isDelActive        := msg(0)(12);
      ddsMsg.isPosValid         := msg(0)(13);
      ddsMsg.currLayerId        := msg(0)(21 downto 14);
      ddsMsg.currSpotId         := msg(1)(3 downto 0) & msg(0)(31 downto 22);
      ddsMsg.posX              	:= msg(1)(16 downto 4);
      ddsMsg.posY             	:= msg(1)(29 downto 17);
      ddsMsg.pad                := msg(1)(31 downto 30);       
     return DdsMsg ;
   end function encDdsMsg;
   ------------------------------------------------------------
 
  ------------------------------------------------------------
   --! ITU DDS  message functions
   function encDdsItuMsg (msg : Slv32Array(14 downto 0); strobe : sl) return ItuMsg2Type is
      variable ItuMsg : ItuMsg2Type;
   begin
      ItuMsg.strobe         := strobe;     
      ItuMsg.statFlagsDds5 := msg(0)(17 downto 15);
      ItuMsg.statFlagsDds4 := msg(0)(14 downto 12);
      ItuMsg.statFlagsDds3 := msg(0)(11 downto 9);
      ItuMsg.statFlagsDds2 := msg(0)(8 downto 6);
      ItuMsg.statFlagsDds1 := msg(0)(5 downto 3);
      ItuMsg.statFlagsDds0 := msg(0)(2 downto 0);      
      
      ItuMsg.dataDds0.currSpotId (9 downto 10) := msg(1)(31 downto 22);
      ItuMsg.dataDds0.currLayerId   := msg(1)(21 downto 14); 
      ItuMsg.dataDds0.isPosValid   := msg(1)(13);
      ItuMsg.dataDds0.isDelActive  := msg(1)(12);
      ItuMsg.dataDds0.msgId        := msg(1)(11 downto 4);
      ItuMsg.dataDds0.ddsInstId    := msg(1)(3 downto 0);
      ItuMsg.dataDds0.currSpotId(13 downto 10) := msg(2)(3 downto 0);
      ItuMsg.dataDds0.posX        := msg(2)(16 downto 4);
      ItuMsg.dataDds0.posY        := msg(2)(29 downto 17);
      ItuMsg.dataDds0.pad         := msg(2)(31 downto 30);
        	        
      ItuMsg.dataDds0.currSpotId (9 downto 10) := msg(1)(31 downto 22);
      ItuMsg.dataDds0.currLayerId   := msg(1)(21 downto 14); 
      ItuMsg.dataDds0.isPosValid   := msg(1)(13);
      ItuMsg.dataDds0.isDelActive  := msg(1)(12);
      ItuMsg.dataDds0.msgId        := msg(1)(11 downto 4);
      ItuMsg.dataDds0.ddsInstId    := msg(1)(3 downto 0);
      ItuMsg.dataDds0.currSpotId(13 downto 10) := msg(2)(3 downto 0);
      ItuMsg.dataDds0.posX        := msg(2)(16 downto 4);
      ItuMsg.dataDds0.posY        := msg(2)(29 downto 17);
      ItuMsg.dataDds0.pad         := msg(2)(31 downto 30);
      
             
      ItuMsg.dataDds1.currSpotId (9 downto 10) := msg(3)(31 downto 22);
      ItuMsg.dataDds1.currLayerId   := msg(3)(21 downto 14); 
      ItuMsg.dataDds1.isPosValid   := msg(3)(13);
      ItuMsg.dataDds1.isDelActive  := msg(3)(12);
      ItuMsg.dataDds1.msgId        := msg(3)(11 downto 4);
      ItuMsg.dataDds1.ddsInstId    := msg(3)(3 downto 0);
      ItuMsg.dataDds1.currSpotId(13 downto 10) := msg(4)(3 downto 0);
      ItuMsg.dataDds1.posX        := msg(4)(16 downto 4);
      ItuMsg.dataDds1.posY        := msg(4)(29 downto 17);
      ItuMsg.dataDds1.pad         := msg(4)(31 downto 30);
                   
      ItuMsg.dataDds2.currSpotId (9 downto 10) := msg(5)(31 downto 22);
      ItuMsg.dataDds2.currLayerId   := msg(5)(21 downto 14); 
      ItuMsg.dataDds2.isPosValid   := msg(5)(13);
      ItuMsg.dataDds2.isDelActive  := msg(5)(12);
      ItuMsg.dataDds2.msgId        := msg(5)(11 downto 4);
      ItuMsg.dataDds2.ddsInstId    := msg(5)(3 downto 0);
      ItuMsg.dataDds2.currSpotId(13 downto 10) := msg(6)(3 downto 0);
      ItuMsg.dataDds2.posX        := msg(6)(16 downto 4);
      ItuMsg.dataDds2.posY        := msg(6)(29 downto 17);
      ItuMsg.dataDds2.pad         := msg(6)(31 downto 30);
 
                   
      ItuMsg.dataDds3.currSpotId (9 downto 10) := msg(7)(31 downto 22);
      ItuMsg.dataDds3.currLayerId   := msg(7)(21 downto 14); 
      ItuMsg.dataDds3.isPosValid   := msg(7)(13);
      ItuMsg.dataDds3.isDelActive  := msg(7)(12);
      ItuMsg.dataDds3.msgId        := msg(7)(11 downto 4);
      ItuMsg.dataDds3.ddsInstId    := msg(7)(3 downto 0);
      ItuMsg.dataDds3.currSpotId(13 downto 10) := msg(8)(3 downto 0);
      ItuMsg.dataDds3.posX        := msg(8)(16 downto 4);
      ItuMsg.dataDds3.posY        := msg(8)(29 downto 17);
      ItuMsg.dataDds3.pad         := msg(8)(31 downto 30);

                   
      ItuMsg.dataDds4.currSpotId (9 downto 10) := msg(9)(31 downto 22);
      ItuMsg.dataDds4.currLayerId   := msg(9)(21 downto 14); 
      ItuMsg.dataDds4.isPosValid   := msg(9)(13);
      ItuMsg.dataDds4.isDelActive  := msg(9)(12);
      ItuMsg.dataDds4.msgId        := msg(9)(11 downto 4);
      ItuMsg.dataDds4.ddsInstId    := msg(9)(3 downto 0);
      ItuMsg.dataDds4.currSpotId(13 downto 10) := msg(10)(3 downto 0);
      ItuMsg.dataDds4.posX        := msg(10)(16 downto 4);
      ItuMsg.dataDds4.posY        := msg(10)(29 downto 17);
      ItuMsg.dataDds4.pad         := msg(10)(31 downto 30);

                   
      ItuMsg.dataDds5.currSpotId (9 downto 10) := msg(11)(31 downto 22);
      ItuMsg.dataDds5.currLayerId   := msg(11)(21 downto 14); 
      ItuMsg.dataDds5.isPosValid   := msg(11)(13);
      ItuMsg.dataDds5.isDelActive  := msg(11)(12);
      ItuMsg.dataDds5.msgId        := msg(11)(11 downto 4);
      ItuMsg.dataDds5.ddsInstId    := msg(11)(3 downto 0);
      ItuMsg.dataDds5.currSpotId(13 downto 10) := msg(12)(3 downto 0);
      ItuMsg.dataDds5.posX        := msg(12)(16 downto 4);
      ItuMsg.dataDds5.posY        := msg(12)(29 downto 17);
      ItuMsg.dataDds5.pad         := msg(12)(31 downto 30);

      ItuMsg.timeStamp(31 downto 0) := msg(13);
      ItuMsg.timeStamp(63 downto 32):= msg(14);
	  
	  return ItuMsg;
	 end function encDdsItuMsg;
	  
   	  --------------------------------------------------------------------------------
   function encHpdTxDdsItu (ituMsg : ItuMsg2Type) return Slv32Array is
      variable hpdTxDdsItu : Slv32Array(14 downto 0);
   begin
      -- Status
      --1 word
      hpdTxDdsItu(0)(31 downto 18):= (others => '0');
      hpdTxDdsItu(0)(17 downto 15):= ituMsg.statFlagsDds5;
      hpdTxDdsItu(0)(14 downto 12):= ituMsg.statFlagsDds4;
      hpdTxDdsItu(0)(11 downto 9) := ituMsg.statFlagsDds3;
      hpdTxDdsItu(0)(8 downto 6)  := ituMsg.statFlagsDds2;
      hpdTxDdsItu(0)(5 downto 3)  := ituMsg.statFlagsDds1;
      hpdTxDdsItu(0)(2 downto 0)  := ituMsg.statFlagsDds0;
      --2 word  - 3 word
	  hpdTxDdsItu(1)(31 downto 22) := ituMsg.dataDds0.currSpotId(9 downto 0);
	  hpdTxDdsItu(1)(21 downto 14) := ituMsg.dataDds0.currLayerId;
	  hpdTxDdsItu(1)(13) := ituMsg.dataDds0.isPosValid;
	  hpdTxDdsItu(1)(12) := ituMsg.dataDds0.isDelActive;
      hpdTxDdsItu(1)(11 downto 4) := ituMsg.dataDds0.msgId;
      hpdTxDdsItu(1)(3 downto 0)  := ituMsg.dataDds0.ddsInstId; 
      hpdTxDdsItu(2)(3 downto 0) := ituMsg.dataDds0.currSpotId(13 downto 10);      
      hpdTxDdsItu(2)(16 downto 4) := ituMsg.dataDds0.posX;   
      hpdTxDdsItu(2)(29 downto 17) := ituMsg.dataDds0.posY;      
      hpdTxDdsItu(2)(31 downto 30) := ituMsg.dataDds0.pad;


      --3 word  - 4 word
	  hpdTxDdsItu(3)(31 downto 22) := ituMsg.dataDds1.currSpotId(9 downto 0);
	  hpdTxDdsItu(3)(21 downto 14) := ituMsg.dataDds1.currLayerId;
	  hpdTxDdsItu(3)(13) := ituMsg.dataDds1.isPosValid;
	  hpdTxDdsItu(3)(12) := ituMsg.dataDds1.isDelActive;
      hpdTxDdsItu(3)(11 downto 4) := ituMsg.dataDds1.msgId;
      hpdTxDdsItu(3)(3 downto 0)  := ituMsg.dataDds1.ddsInstId; 
      hpdTxDdsItu(4)(3 downto 0) := ituMsg.dataDds1.currSpotId(13 downto 10);      
      hpdTxDdsItu(4)(16 downto 4) := ituMsg.dataDds1.posX;   
      hpdTxDdsItu(4)(29 downto 17) := ituMsg.dataDds1.posY;      
      hpdTxDdsItu(4)(31 downto 30) := ituMsg.dataDds1.pad;


--5 word  - 6 word
	  hpdTxDdsItu(5)(31 downto 22) := ituMsg.dataDds2.currSpotId(9 downto 0);
	  hpdTxDdsItu(5)(21 downto 14) := ituMsg.dataDds2.currLayerId;
	  hpdTxDdsItu(5)(13) := ituMsg.dataDds2.isPosValid;
	  hpdTxDdsItu(5)(12) := ituMsg.dataDds2.isDelActive;
      hpdTxDdsItu(5)(11 downto 4) := ituMsg.dataDds2.msgId;
      hpdTxDdsItu(5)(3 downto 0)  := ituMsg.dataDds2.ddsInstId; 
      hpdTxDdsItu(6)(3 downto 0) := ituMsg.dataDds2.currSpotId(13 downto 10);      
      hpdTxDdsItu(6)(16 downto 4) := ituMsg.dataDds2.posX;   
      hpdTxDdsItu(6)(29 downto 17) := ituMsg.dataDds2.posY;      
      hpdTxDdsItu(6)(31 downto 30) := ituMsg.dataDds2.pad;


--7 word  - 8 word
	  hpdTxDdsItu(7)(31 downto 22) := ituMsg.dataDds3.currSpotId(9 downto 0);
	  hpdTxDdsItu(7)(21 downto 14) := ituMsg.dataDds3.currLayerId;
	  hpdTxDdsItu(7)(13) := ituMsg.dataDds3.isPosValid;
	  hpdTxDdsItu(7)(12) := ituMsg.dataDds3.isDelActive;
      hpdTxDdsItu(7)(11 downto 4) := ituMsg.dataDds3.msgId;
      hpdTxDdsItu(7)(3 downto 0)  := ituMsg.dataDds3.ddsInstId; 
      hpdTxDdsItu(8)(3 downto 0) := ituMsg.dataDds3.currSpotId(13 downto 10);      
      hpdTxDdsItu(8)(16 downto 4) := ituMsg.dataDds3.posX;   
      hpdTxDdsItu(8)(29 downto 17) := ituMsg.dataDds3.posY;      
      hpdTxDdsItu(8)(31 downto 30) := ituMsg.dataDds3.pad;


--9 word  - 10 word
	  hpdTxDdsItu(9)(31 downto 22) := ituMsg.dataDds4.currSpotId(9 downto 0);
	  hpdTxDdsItu(9)(21 downto 14) := ituMsg.dataDds4.currLayerId;
	  hpdTxDdsItu(9)(13) := ituMsg.dataDds4.isPosValid;
	  hpdTxDdsItu(9)(12) := ituMsg.dataDds4.isDelActive;
      hpdTxDdsItu(9)(11 downto 4) := ituMsg.dataDds4.msgId;
      hpdTxDdsItu(9)(3 downto 0)  := ituMsg.dataDds4.ddsInstId; 
      hpdTxDdsItu(10)(3 downto 0) := ituMsg.dataDds4.currSpotId(13 downto 10);      
      hpdTxDdsItu(10)(16 downto 4) := ituMsg.dataDds4.posX;   
      hpdTxDdsItu(10)(29 downto 17) := ituMsg.dataDds4.posY;      
      hpdTxDdsItu(10)(31 downto 30) := ituMsg.dataDds4.pad;


--11 word  - 12 word
	  hpdTxDdsItu(11)(31 downto 22) := ituMsg.dataDds5.currSpotId(9 downto 0);
	  hpdTxDdsItu(11)(21 downto 14) := ituMsg.dataDds5.currLayerId;
	  hpdTxDdsItu(11)(13) := ituMsg.dataDds5.isPosValid;
	  hpdTxDdsItu(11)(12) := ituMsg.dataDds5.isDelActive;
      hpdTxDdsItu(11)(11 downto 4) := ituMsg.dataDds5.msgId;
      hpdTxDdsItu(11)(3 downto 0)  := ituMsg.dataDds5.ddsInstId; 
      hpdTxDdsItu(12)(3 downto 0) := ituMsg.dataDds5.currSpotId(13 downto 10);      
      hpdTxDdsItu(12)(16 downto 4) := ituMsg.dataDds5.posX;   
      hpdTxDdsItu(12)(29 downto 17) := ituMsg.dataDds5.posY;      
      hpdTxDdsItu(12)(31 downto 30) := ituMsg.dataDds5.pad;


      -- Timestamp
      hpdTxDdsItu(13)(31 downto 0) := ituMsg.timeStamp(31 downto 0);
      hpdTxDdsItu(14)(31 downto 0) := ituMsg.timeStamp(63 downto 32);
      return hpdTxDdsItu ;
   end function encHpdTxDdsItu; 
 
 -------------------------------------------------------------------
end package body CmuCorePkg;
---------------------------------------------------------------------------------------------------