---------------------------------------------------------------------------------------------------
--! @brief     Pulse train generator module
--! @author Uros Legat, Cosylab (uros.legat@cosylab.com)
--!
--! @date April 10 2019 created
--!
--! @version v1.0
--!
--!
--! @file PulseTrGen.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.numeric_std.all;
use ieee.std_logic_arith.all;
use work.CslStdRtlPkg.all;

--! @brief  Pulse train generator module
--! @details The when enabled module generates a pulse train of adjustable period and pulse width.
--!          period_i - A period of 0 is invalid.
--!              Example: period_i = 0,  pulse_o disabled constant '0'
--!              Example: period_i = 1,  pulse_o clk_i div by 2
--!              Example: period_i = 2,  pulse_o clk_i div by 3
--!          width_i - A width of 0 is invalid.
--!              Example: width_i = 0,  pulse_o constant '0'
--!              Example: width_i = 1,  pulse_o width 1
--!              Example: width_i = 2,  pulse_o width 2
--!          The module increments the period and width counters only if enabled.
--!          A periodic en_i can also serve as a further period divider.
--!          
--! @author Uros Legat, Cosylab (uros.legat@cosylab.com)
---------------------------------------------------------------------------------------------------
entity PulseTrGen is
   generic (
      TPD_G             : time      := 1 ns;
      WIDTH_G           : positive  := 32
      );
   port(
      clk_i          : in  sl;   --! input clock
      rst_i          : in  sl;   --! input reset
      en_i           : in  sl:='1'; --! Count enable (can be used to divide the period and width)
      period_i       : in  slv(WIDTH_G-1 downto 0); --! Pulse period
      width_i        : in  slv(WIDTH_G-1 downto 0):= toSlv(1,WIDTH_G); --! Pulse width '1' by default
      pulse_o        : out sl;   --! Output pulse
      tick_o         : out sl    --! Output tick (width of 1 c-c)
   );
end entity PulseTrGen;

architecture rtl of PulseTrGen is
   --! Record containing all register elements
   type RegType is record
      en  : sl;
   end record RegType;
   
   --! Initial and reset values for all register elements
   constant REG_INIT_C : RegType := (
      en => '0');
   
   --! Output of registers
   signal r          : RegType := REG_INIT_C;
   --! Combinatorial input to registers
   signal rin        : RegType;
   
   --! internal signal
   signal perTout  : sl;
   signal widthTout  : sl;
   
begin

   --! Period watchdog counter
   u0_period: entity work.Watchdog
   generic map (
      TPD_G   => TPD_G,
      WIDTH_G => WIDTH_G)
   port map (
      clk_i   => clk_i,
      rst_i   => rst_i,
      en_i    => en_i,
      wdRst_i => perTout,
      time_i  => period_i,
      tout_o  => perTout);
      
      
   --! Pulse width watchdog counter
   u1_width: entity work.Watchdog
   generic map (
      TPD_G   => TPD_G,
      WIDTH_G => WIDTH_G)
   port map (
      clk_i   => clk_i,
      rst_i   => rst_i,
      en_i    => r.en,
      wdRst_i => perTout,
      time_i  => width_i,
      tout_o  => widthTout);    
   
   --! Edge detection component for tick generation
   u_tick : entity work.CslEdgeDet 
      generic map(
         TPD_G => TPD_G,
         FALLING_EDGE_G => FALSE
      )
      port map(
         clk_i => clk_i,
         rst_i => rst_i,
         sig_i => widthTout,
         sig_o => tick_o
      );

   --! @brief Combinational process
   comb : process (r, rst_i, en_i) is 
      variable v : RegType;
   begin
      v := r; -- default assignment
      
      v.en := en_i;
      
      if (rst_i = '1') then -- reset condition
         v := REG_INIT_C;
      end if;

      rin  <= v; -- drive register inputs
   ---------------------------------------------------
   end process comb;
   
   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk to create registers
   --! @param[in]  rin, clk_i
   --! @param[out] r 
   seq : process (clk_i) is -- sequential process
   begin
      if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
      end if;
   end process seq;
   ---
   pulse_o <= not widthTout;
   
end architecture rtl;
---------------------------------------------------------------------------------------------------