---------------------------------------------------------------------------------------------------
--! @brief  SFED Message concentrator
--! @details    
--!      The message concentrator receives messages from the 12 SFEDs (sfedMsg_i), processes the messages 
--!      and the internal statuses, and combines the relevant information into the (ituMsg_o).
--!      
--!      Note: The ITU message strobe and the SFED message data are not synced. 
--!      Registers the data and produces 1 c-c delay.
--!    
--! @author Uros Legat, Cosylab (uros.legat@cosylab.com)
--!
--! @date Oct 07 2019 created
--! @date Dec 20 2019 last modify
--!
--! @version v1.0
--!
--! @file SfedMsgConc.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
--
use work.CmuCorePkg.all;
---------------------------------------------------------------------------------------------------
entity SfedMsgConc is
   generic (
      TPD_G  : time    := 1 ns;
      N_BL_G : positive:= 6;
      ITU_PERIOD_G : positive:= 2500
   );
   port(
      clk_i       : in  sl;
      rst_i      : in  sl;

      -- SFED messages and statuses
      sfedMsg_i   : in SfedMsgArray(N_BL_G*2-1 downto 0);
      sfedStat_i  : in SfedStatArray(N_BL_G*2-1 downto 0);
      fecCtrl_i   : in FecCtrlType;      
      fsmStat_i   : in FsmStatType;
      rstTs_i     : in sl;
      --
      ituMsg_o    : out ItuMsgType;
      stobe_o     : out sl;  
      sfedErr_o   : out sl;
      sfedLink_o  : out sl
   );
end SfedMsgConc;
---------------------------------------------------------------------------------------------------
architecture rtl of SfedMsgConc is

   --! Record containing all register elements 
   type RegType is record
      ituMsgBuff    : ItuMsgType;
      counter       : unsigned(63 downto 0);
      sfedLinks     : slv(N_BL_G*2-1 downto 0);
      en            : sl;
   end record RegType;

   --! Initial and reset values for all register elements
   constant REG_INIT_C : RegType := (
      ituMsgBuff    => ITU_MSG_INIT_C,
      counter       => (others => '0'),
      sfedLinks     => (others => '0'),
      en            => '0');

   --! Internal signal
   signal strobe : sl;
   
   --! Output of registers          
   signal r : RegType := REG_INIT_C;

   --! Combinatorial input to registers
   signal rin : RegType;
---------------------------------------------------------------------------------------------------
begin

   u_PulseTrGen: entity work.PulseTrGen
      generic map (
         TPD_G   => TPD_G,
         WIDTH_G => 16)
      port map (
         clk_i    => clk_i,
         rst_i    => rst_i,
         en_i     => r.en,
         period_i => toSlv(ITU_PERIOD_G-1,16),
         tick_o   => strobe);

     --! @brief Combinational process
   --! @details Main module logic. Generates rin based on r and any module inputs
   comb : process (r, rst_i, strobe, sfedMsg_i, sfedStat_i, fecCtrl_i, fsmStat_i, rstTs_i) is
      variable v : RegType;
     
   begin
      -- Register the current value
      v  := r;
      
      -- Data path
      --------------------------------------------------------------------------------------------
      -- Only relative currents are concentrated into the ITU message
      for i in 0 to N_BL_G-1 loop
         if (fecCtrl_i.sfedEnable(i)='1') then
            -- X
            v.ituMsgBuff.setpAbsCurrX(i)  := sfedMsg_i(2*i).setpAbsCurr; 
            v.ituMsgBuff.measAbsCurrX(i)  := sfedMsg_i(2*i).measAbsCurr;
            v.ituMsgBuff.setpRelCurrX(i)  := sfedMsg_i(2*i).setpRelCurr; 
            v.ituMsgBuff.measRelCurrX(i)  := sfedMsg_i(2*i).measRelCurr;    
            v.ituMsgBuff.timeStamp1X(i)   := sfedMsg_i(2*i).timestamp(31 downto 0);                                      
            v.ituMsgBuff.timeStamp2X(i)   := sfedMsg_i(2*i).timestamp(63 downto 32);
            -- Y
            v.ituMsgBuff.setpAbsCurrY(i)  := sfedMsg_i(2*i+1).setpAbsCurr; 
            v.ituMsgBuff.measAbsCurrY(i)  := sfedMsg_i(2*i+1).measAbsCurr;
            v.ituMsgBuff.setpRelCurrY(i)  := sfedMsg_i(2*i+1).setpRelCurr; 
            v.ituMsgBuff.measRelCurrY(i)  := sfedMsg_i(2*i+1).measRelCurr;         
            v.ituMsgBuff.timeStamp1Y(i)   := sfedMsg_i(2*i).timestamp(31 downto 0);                                      
            v.ituMsgBuff.timeStamp2Y(i)   := sfedMsg_i(2*i).timestamp(63 downto 32);    
           
         else
            -- X
            v.ituMsgBuff.setpAbsCurrX(i)  := (others => '0');
            v.ituMsgBuff.measAbsCurrX(i)  := (others => '0');
            v.ituMsgBuff.setpRelCurrX(i)  := (others => '0');
            v.ituMsgBuff.measRelCurrX(i)  := (others => '0');
            v.ituMsgBuff.timeStamp1X(i)   := (others => '0');
            v.ituMsgBuff.timeStamp2X(i)   := (others => '0');
            -- Y                          
            v.ituMsgBuff.setpAbsCurrY(i)  := (others => '0');
            v.ituMsgBuff.measAbsCurrY(i)  := (others => '0');
            v.ituMsgBuff.setpRelCurrY(i)  := (others => '0');
            v.ituMsgBuff.measRelCurrY(i)  := (others => '0');
            v.ituMsgBuff.timeStamp1Y(i)   := (others => '0');
            v.ituMsgBuff.timeStamp2Y(i)   := (others => '0');
         end if;
      end loop;


      -- Status path
      --------------------------------------------------------------------------------------------     
--      for i in 0 to N_BL_G-1 loop
--         if (fecCtrl_i.sfedEnable(i)='1') then
--            v.ituMsgBuff.statFlags(2*i)   := sfedStat_i(2*i).err;   
--            v.ituMsgBuff.statFlags(2*i+1) := sfedStat_i(2*i+1).err;
--            v.sfedLinks(2*i)              := sfedStat_i(2*i).link;
--            v.sfedLinks(2*i+1)            := sfedStat_i(2*i+1).link;            
--         else
--            v.ituMsgBuff.statFlags(2*i)   := '0';
--            v.ituMsgBuff.statFlags(2*i+1) := '0';
--            v.sfedLinks(2*i)              := '1';
--            v.sfedLinks(2*i+1)            := '1';
--         end if;
--      end loop;      

--    for i in 0 to N_BL_G-1 loop      
--        v.ituMsgBuff.statFlagsX(i)(2)   := uOr(sfedStat_i(2*i).crcCnt)or uOr(sfedStat_i(2*i).lenCnt)or uOr(sfedStat_i(2*i).dropCnt);   
--        v.ituMsgBuff.statFlagsX(i)(1)   := uOr(sfedStat_i(2*i).gtDecErrCnt) or uOr(sfedStat_i(2*i).gtDispErrCnt) or uOr(sfedStat_i(2*i).gtByteReAlig); 
--	    v.ituMsgBuff.statFlagsX(i)(0)	:= uOr(sfedStat_i(2*i).tmstpStallCnt) or uOr(sfedStat_i(2*i).toutCnt) ;
	   
--	    v.ituMsgBuff.statFlagsY(i)(2)   := uOr(sfedStat_i(2*i+1).crcCnt)or uOr(sfedStat_i(2*i+1).lenCnt)or uOr(sfedStat_i(2*i+1).dropCnt);   
--        v.ituMsgBuff.statFlagsY(i)(1)   := uOr(sfedStat_i(2*i+1).gtDecErrCnt) or uOr(sfedStat_i(2*i+1).gtDispErrCnt) or uOr(sfedStat_i(2*i+1).gtByteReAlig); 
--	    v.ituMsgBuff.statFlagsY(i)(0)	:= uOr(sfedStat_i(2*i+1).tmstpStallCnt) or uOr(sfedStat_i(2*i+1).toutCnt) ;
--	end loop;
          
        v.ituMsgBuff.statFlagsX0(2)   := uOr(sfedStat_i(0).crcCnt)or uOr(sfedStat_i(0).lenCnt)or uOr(sfedStat_i(0).dropCnt);   
        v.ituMsgBuff.statFlagsX0(1)   := uOr(sfedStat_i(0).gtDecErrCnt) or uOr(sfedStat_i(0).gtDispErrCnt) or uOr(sfedStat_i(0).gtByteReAlig); 
	    v.ituMsgBuff.statFlagsX0(0)	:= uOr(sfedStat_i(0).tmstpStallCnt) or uOr(sfedStat_i(0).toutCnt) ; 
	    v.ituMsgBuff.statFlagsY0(2)   := uOr(sfedStat_i(1).crcCnt)or uOr(sfedStat_i(1).lenCnt)or uOr(sfedStat_i(1).dropCnt);   
        v.ituMsgBuff.statFlagsY0(1)   := uOr(sfedStat_i(1).gtDecErrCnt) or uOr(sfedStat_i(1).gtDispErrCnt) or uOr(sfedStat_i(1).gtByteReAlig); 
	    v.ituMsgBuff.statFlagsY0(0)	:= uOr(sfedStat_i(1).tmstpStallCnt) or uOr(sfedStat_i(1).toutCnt) ;

        v.ituMsgBuff.statFlagsX1(2)   := uOr(sfedStat_i(2).crcCnt)or uOr(sfedStat_i(2).lenCnt)or uOr(sfedStat_i(2).dropCnt);   
        v.ituMsgBuff.statFlagsX1(1)   := uOr(sfedStat_i(2).gtDecErrCnt) or uOr(sfedStat_i(2).gtDispErrCnt) or uOr(sfedStat_i(2).gtByteReAlig); 
	    v.ituMsgBuff.statFlagsX1(0)	:= uOr(sfedStat_i(2).tmstpStallCnt) or uOr(sfedStat_i(2).toutCnt) ; 
	    v.ituMsgBuff.statFlagsY1(2)   := uOr(sfedStat_i(3).crcCnt)or uOr(sfedStat_i(3).lenCnt)or uOr(sfedStat_i(3).dropCnt);   
        v.ituMsgBuff.statFlagsY1(1)   := uOr(sfedStat_i(3).gtDecErrCnt) or uOr(sfedStat_i(3).gtDispErrCnt) or uOr(sfedStat_i(3).gtByteReAlig); 
	    v.ituMsgBuff.statFlagsY1(0)	:= uOr(sfedStat_i(3).tmstpStallCnt) or uOr(sfedStat_i(3).toutCnt) ;
	    
	    v.ituMsgBuff.statFlagsX2(2)   := uOr(sfedStat_i(4).crcCnt)or uOr(sfedStat_i(4).lenCnt)or uOr(sfedStat_i(4).dropCnt);   
        v.ituMsgBuff.statFlagsX2(1)   := uOr(sfedStat_i(4).gtDecErrCnt) or uOr(sfedStat_i(4).gtDispErrCnt) or uOr(sfedStat_i(4).gtByteReAlig); 
	    v.ituMsgBuff.statFlagsX2(0)	:= uOr(sfedStat_i(4).tmstpStallCnt) or uOr(sfedStat_i(4).toutCnt) ; 
	    v.ituMsgBuff.statFlagsY2(2)   := uOr(sfedStat_i(5).crcCnt)or uOr(sfedStat_i(5).lenCnt)or uOr(sfedStat_i(5).dropCnt);   
        v.ituMsgBuff.statFlagsY2(1)   := uOr(sfedStat_i(5).gtDecErrCnt) or uOr(sfedStat_i(5).gtDispErrCnt) or uOr(sfedStat_i(5).gtByteReAlig); 
	    v.ituMsgBuff.statFlagsY2(0)	:= uOr(sfedStat_i(5).tmstpStallCnt) or uOr(sfedStat_i(5).toutCnt) ;
	    
	    v.ituMsgBuff.statFlagsX3(2)   := uOr(sfedStat_i(6).crcCnt)or uOr(sfedStat_i(6).lenCnt)or uOr(sfedStat_i(6).dropCnt);   
        v.ituMsgBuff.statFlagsX3(1)   := uOr(sfedStat_i(6).gtDecErrCnt) or uOr(sfedStat_i(6).gtDispErrCnt) or uOr(sfedStat_i(6).gtByteReAlig); 
	    v.ituMsgBuff.statFlagsX3(0)	:= uOr(sfedStat_i(6).tmstpStallCnt) or uOr(sfedStat_i(6).toutCnt) ; 
	    v.ituMsgBuff.statFlagsY3(2)   := uOr(sfedStat_i(7).crcCnt)or uOr(sfedStat_i(7).lenCnt)or uOr(sfedStat_i(7).dropCnt);   
        v.ituMsgBuff.statFlagsY3(1)   := uOr(sfedStat_i(7).gtDecErrCnt) or uOr(sfedStat_i(7).gtDispErrCnt) or uOr(sfedStat_i(7).gtByteReAlig); 
	    v.ituMsgBuff.statFlagsY3(0)	:= uOr(sfedStat_i(7).tmstpStallCnt) or uOr(sfedStat_i(7).toutCnt) ;
	    
	    v.ituMsgBuff.statFlagsX4(2)   := uOr(sfedStat_i(8).crcCnt)or uOr(sfedStat_i(8).lenCnt)or uOr(sfedStat_i(8).dropCnt);   
        v.ituMsgBuff.statFlagsX4(1)   := uOr(sfedStat_i(8).gtDecErrCnt) or uOr(sfedStat_i(8).gtDispErrCnt) or uOr(sfedStat_i(8).gtByteReAlig); 
	    v.ituMsgBuff.statFlagsX4(0)	:= uOr(sfedStat_i(8).tmstpStallCnt) or uOr(sfedStat_i(8).toutCnt) ; 
	    v.ituMsgBuff.statFlagsY4(2)   := uOr(sfedStat_i(9).crcCnt)or uOr(sfedStat_i(9).lenCnt)or uOr(sfedStat_i(9).dropCnt);   
        v.ituMsgBuff.statFlagsY4(1)   := uOr(sfedStat_i(9).gtDecErrCnt) or uOr(sfedStat_i(9).gtDispErrCnt) or uOr(sfedStat_i(9).gtByteReAlig); 
	    v.ituMsgBuff.statFlagsY4(0)	:= uOr(sfedStat_i(9).tmstpStallCnt) or uOr(sfedStat_i(9).toutCnt) ;
	    
	    v.ituMsgBuff.statFlagsX5(2)   := uOr(sfedStat_i(10).crcCnt)or uOr(sfedStat_i(10).lenCnt)or uOr(sfedStat_i(10).dropCnt);   
        v.ituMsgBuff.statFlagsX5(1)   := uOr(sfedStat_i(10).gtDecErrCnt) or uOr(sfedStat_i(10).gtDispErrCnt) or uOr(sfedStat_i(10).gtByteReAlig); 
	    v.ituMsgBuff.statFlagsX5(0)	:= uOr(sfedStat_i(10).tmstpStallCnt) or uOr(sfedStat_i(10).toutCnt) ; 
	    v.ituMsgBuff.statFlagsY5(2)   := uOr(sfedStat_i(11).crcCnt)or uOr(sfedStat_i(11).lenCnt)or uOr(sfedStat_i(11).dropCnt);   
        v.ituMsgBuff.statFlagsY5(1)   := uOr(sfedStat_i(11).gtDecErrCnt) or uOr(sfedStat_i(11).gtDispErrCnt) or uOr(sfedStat_i(11).gtByteReAlig); 
	    v.ituMsgBuff.statFlagsY5(0)	:= uOr(sfedStat_i(11).tmstpStallCnt) or uOr(sfedStat_i(11).toutCnt) ;
	    



      -- Enable timestamp and message strobe
      --------------------------------------------------------------------------------------------
      if (fsmStat_i.state = OPER_C or fsmStat_i.state = FAIL_C) then
         v.en := '1';
      else
         v.en := '0';      
      end if;
      
      -- Timestamp counter
      --------------------------------------------------------------------------------------------
      if (r.en = '1') then
         v.counter := r.counter + 1;
      end if;
      
      -- Reset/clear the counter if disabled or requested by FEC in TIMESYNC message
      if (r.en = '0' or rstTs_i = '1') then      
         v.counter := (others => '0');  
      end if;

      --
      v.ituMsgBuff.timeStamp := slv(r.counter);
      
      -- Strobe
      --------------------------------------------------------------------------------------------
      v.ituMsgBuff.strobe := strobe;
      
      -- Reset
      --------------------------------------------------------------------------------------------
      if (rst_i = '1') then
         v := REG_INIT_C;
      end if;

      -- Register the variable for next clock cycle
      rin <= v;
      
      -- Output 
      ituMsg_o <= r.ituMsgBuff;
      stobe_o <= strobe;
      -- Logic OR of all the enabled SFED statuses
     -- sfedErr_o <= uor(r.ituMsgBuff.statFlags);
      sfedLink_o <= uand(r.sfedLinks);
   end process comb;

   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk_i_i to create registers
   seq : process (clk_i) is
   begin
      if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
      end if;
   end process seq;

end rtl;
---------------------------------------------------------------------------------------------------