---------------------------------------------------------------------------------------------------
--! @brief  FEC link module used in HDL simulation and CMU simulator
--!     
--! @author Uros Legat, Cosylab (uros.legat@cosylab.com)
--!
--! @date Oct 07 2019 created
--! @date Dec 20 2019 last modify
--!
--! @version v1.0
--!
--! @file SimItuLink.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
use work.CmuCorePkg.all;
use work.FiberPkg.all;
---------------------------------------------------------------------------------------------------
entity SimFecLink is
   generic (
      TPD_G       : time     := 1 ns;
      LINK_TIME_G : positive := 100000; --! link time generic for FiberMon
    --  WD_TIME_G   : positive := 2500);  --! watchdog timer generic     
      WD_TIME_G   : positive := 500);  --! watchdog timer generic
   port (
      clk_i : in sl; --! clock signal bus
      rst_i : in sl; --! reset signal bus

      -- Configuration interface
      cfgStrobe_i   : in  sl                      := '0';
      cfgRequest_i  : in  Slv32Array(23 downto 0) := (others => x"00000000");
      cfgResponse_o : out Slv32Array(23 downto 0);
      cfgDrop_o     : out sl;

      gtRx_i : in  GtType; --! GT Rx signals
      gtTx_o : out GtType; --! GT Rx signals

      -- ITU message reception
      clear_i   : in  sl;           --! clear counters signal
      ituStat_o : out SfedStatType; --! CMU stat signals
      ilCom_o   : out sl;           --! interlock signal from FiberMon
      ituMsg_o  : out ItuMsgType;   --! CMU message
                                    --
      rstTs_i : in sl := '0'        --! reset the timestamp (send the timesync message)  
   );
end SimFecLink;
---------------------------------------------------------------------------------------------------
architecture structure of SimFecLink is

   --! Size of the message in 32-bit words
   constant ITU_MSG_SIZE_C : positive := 15;
   constant ITU_SFED_MSG_SIZE_C : positive := 22;
   
   --! Fiber Monitor
   signal link_po    : sl              := '1';
   signal statusBits : slv(5 downto 0) := (others => '0');
   signal ituStat    : SfedStatType;

   --! configuration message
   signal ituMsg : ituMsgType;

   --! RX
   signal hpdMsg32array   : Slv32Array(44 downto 0);
   signal hpdRxAxisMaster : AxiStreamMasterType := AXI_STREAM_MASTER_INIT_C;
   signal hpdRxAxisSlave  : AxiStreamSlaveType;
   signal lpdRxAxisMaster : AxiStreamMasterType := AXI_STREAM_MASTER_INIT_C;
   signal lpdRxAxisSlave  : AxiStreamSlaveType;
   -- TX
   signal lpdTxAxisMaster : AxiStreamMasterType := AXI_STREAM_MASTER_INIT_C;
   signal lpdTxAxisSlave  : AxiStreamSlaveType  := AXI_STREAM_SLAVE_INIT_C;
   signal lpdIdle         : sl;
   signal hpdTxMsg32Array : Slv32Array(4-1 downto 0);
   signal hpdTxAxisMaster : AxiStreamMasterType := AXI_STREAM_MASTER_INIT_C;
   signal hpdTxAxisSlave  : AxiStreamSlaveType;
   signal hpdTxPktFull    : sl;
   signal hpdIdle         : sl;

   --! Output message
   signal hpdStrobe : sl;
   signal hpdCrcErr : sl;
   signal hpdLenErr : sl;

   signal tmstpStall : sl;
   signal tout       : sl;
   signal state      : sl;
   signal wdRst      : sl;
   signal cmuDrop    : sl;

   --! CmuLink record
   type RegType is record
      tmstpStall   : sl;
      timeStampReg : slv(63 downto 0);
   end record RegType;

   --! Initial and reset values for all register elements
   constant REG_INIT_C : RegType := (
         tmstpStall   => '0',
         timeStampReg => (others => '0'));

   --! Output of registers          
   signal r : RegType := REG_INIT_C;

   --! Combinatorial input to registers
   signal rin : RegType;
---------------------------------------------------------------------------------------------------  
begin
   ------------------------------------------------------------------------------------------------
   -- Configuration interface on LPD
   ------------------------------------------------------------------------------------------------
   ------------------------------------------------------------------------------------------------
   -- Configuration request
   ------------------------------------------------------------------------------------------------   
   u0_MsgStreamTx : entity work.MsgStreamTx
      generic map (
         TPD_G  => TPD_G,
         SIZE_G => 24)
      port map (
         clk_i        => clk_i,
         rst_i        => rst_i,
         strobe_i     => cfgStrobe_i,
         msg32Array_i => cfgRequest_i,
         full_i       => '0',
         axisMaster_o => lpdTxAxisMaster,
         axisSlave_i  => lpdTxAxisSlave,
         drop_o       => open);
   ------------------------------------------------------------------------------------------------    
   --! Send TIMESYNC message
   ------------------------------------------------------------------------------------------------
   hpdTxMsg32Array(0) <= x"0000_0001";
   hpdTxMsg32Array(1) <= x"0000_0000";
   hpdTxMsg32Array(2) <= x"0000_0000";
   hpdTxMsg32Array(3) <= x"0000_0000";

   u1_MsgStreamTx : entity work.MsgStreamTx
      generic map (
         TPD_G  => TPD_G,
         SIZE_G => 4)
      port map (
         clk_i        => clk_i,
         rst_i        => rst_i,
         strobe_i     => rstTs_i,
         msg32Array_i => hpdTxMsg32Array,
         full_i       => hpdTxPktFull,
         axisMaster_o => hpdTxAxisMaster,
         axisSlave_i  => hpdTxAxisSlave,
         drop_o       => open);

   --! FiberLinkTx module to transmit the messages over GT fiber 
   u0_FiberLinkMapTx : entity work.FiberLinkMapTx
      generic map (
         TPD_G            => TPD_G,
         HPD_DATA_WIDTH_G => 4,
         HPD_MAX_SIZE_G   => 4,
         LPD_DATA_WIDTH_G => 4,
         LPD_PIPES_G      => 1,
         LPD_MAX_SIZE_G   => 24)
      port map (
         clk_i                => clk_i,
         rst_i                => rst_i,
         hpdTxAxisMaster_i    => hpdTxAxisMaster,
         hpdTxAxisSlave_o     => hpdTxAxisSlave,
         hpdTxPktFull_o       => hpdTxPktFull,
         hpdTxErr_o           => open,
         hpdIdle_o            => hpdIdle,
         lpdTxAxisMaster_i(0) => lpdTxAxisMaster,
         lpdTxAxisSlave_o(0)  => lpdTxAxisSlave,
         lpdTxPktFull_o       => open,
         lpdTxErr_o           => open,
         lpdIdle_o            => lpdIdle,
         gtDataTx_o           => gtTx_o.data,
         gtCharTx_o           => gtTx_o.char);

   --! driving Clock correction enable signal from Lpd and Hpd Idle signals
   gtTx_o.clkCorrEn <= lpdIdle and hpdIdle;

   ------------------------------------------------------------------------------------------------
   -- Response
   ------------------------------------------------------------------------------------------------   
   u0_FiberLinkRx : entity work.FiberLinkRx
      generic map (
         TPD_G        => TPD_G,
         DATA_WIDTH_G => 4,
         PIPES_G      => 1,
         IDLE_G       => '0' & x"00",
         SOF_CHAR_G   => x"1C",
         EOF_CHAR_G   => x"FE",
         MAX_SIZE_G   => 24
      )
      port map (
         clk_i           => clk_i,
         rst_i           => rst_i,
         dataRx_i        => gtRx_i.data(7 downto 0),
         charRx_i        => gtRx_i.char(0),
         link_i          => ituStat.link,
         axisMaster_o(0) => lpdRxAxisMaster,
         axisSlave_i(0)  => lpdRxAxisSlave,
         pktEmpty_o      => open,
         drop_o          => cfgDrop_o
      );

   u0_MsgStreamRx : entity work.MsgStreamRx
      generic map (
         TPD_G      => TPD_G,
         SIZE_G     => 24,
         REGISTER_G => true)
      port map (
         clk_i        => clk_i,
         rst_i        => rst_i,
         axisMaster_i => lpdRxAxisMaster,
         axisSlave_o  => lpdRxAxisSlave,
         strobe_o     => open,
         msg32Array_o => cfgResponse_o,
         crcErr_o     => open,
         lenErr_o     => open);

   ------------------------------------------------------------------------------------------------
   -- Reception of ITU messages on HPD
   ------------------------------------------------------------------------------------------------

   --! watchdog reset
   wdRst <= hpdStrobe or clear_i or tout;

   --! Watchdog module instantiation
   u0_Watchdog : entity work.Watchdog
      generic map (
         TPD_G   => TPD_G,
         WIDTH_G => bitSize(WD_TIME_G)
      )
      port map (
         clk_i   => clk_i,
         rst_i   => rst_i,
         en_i    => '1',
         wdRst_i => wdRst,
         time_i  => toSlv(WD_TIME_G, bitSize(WD_TIME_G)),
         tout_o  => tout
      );

   --! Fiber Link Rx module instantiation
   u1_FiberLinkRx : entity work.FiberLinkRx
      generic map (
         TPD_G        => TPD_G,
         DATA_WIDTH_G => 4,
         PIPES_G      => 1,
         IDLE_G       => '0' & x"00",
         SOF_CHAR_G   => x"1C",
         EOF_CHAR_G   => x"FE",
         MAX_SIZE_G   => 45
      )
      port map (
         clk_i           => clk_i,
         rst_i           => rst_i,
         dataRx_i        => gtRx_i.data(15 downto 8),
         charRx_i        => gtRx_i.char(1),
         link_i          => ituStat.link,
         axisMaster_o(0) => hpdRxAxisMaster,
         axisSlave_i(0)  => hpdRxAxisSlave,
         pktEmpty_o      => open,
         drop_o          => cmuDrop
      );

   --! Message Stream Rx module instantiation
   u1_MsgStreamRx : entity work.MsgStreamRx
      generic map (
         TPD_G      => TPD_G,
         SIZE_G     => 45,
         REGISTER_G => true)
      port map (
         clk_i        => clk_i,
         rst_i        => rst_i,
         axisMaster_i => hpdRxAxisMaster,
         axisSlave_o  => hpdRxAxisSlave,
         strobe_o     => hpdStrobe,
         msg32Array_o => hpdMsg32array,
         crcErr_o     => hpdCrcErr,
         lenErr_o     => hpdLenErr);

   --! CMU message encoding from 32 bit vector array
   ituMsg <= encFecMsg(hpdMsg32array, hpdStrobe);

   --! Assign status signals to StatusBits vector
   statusBits(0) <= hpdCrcErr;             -- RX CMU CRC errors
   statusBits(1) <= hpdLenErr;             -- RX CMU LEN errors
   statusBits(2) <= cmuDrop;               -- RX CMU drop errors
   statusBits(3) <= r.tmstpStall;          -- timestamp stall
   statusBits(4) <= tout;                  -- timeout counter
  -- statusBits(5) <= uOr(ituMsg.statFlags); -- Cmu Status flags unary OR

   --! Fiber Monitor module instantiation
   u_FiberMon : entity work.FiberMon
      generic map (
         TPD_G             => TPD_G,
         STAT_SIZE_G       => 6,
         LINK_TIME_WIDTH_G => bitSize(LINK_TIME_G)
      )
      port map (
         clk_i             => clk_i,
         rst_i             => rst_i,
         clear_i           => clear_i,
         statusBits_i      => statusBits,
         gtRx_i            => gtRx_i,
         linkTime_i        => toSlv(LINK_TIME_G, bitSize(LINK_TIME_G)),
         ilk_o             => ilCom_o,
         err_o             => ituStat.err,
         sts32array_o(0)   => ituStat.gtCdrStableCnt,
         sts32array_o(1)   => ituStat.gtDecErrCnt,
         sts32array_o(2)   => ituStat.gtDispErrCnt,
         sts32array_o(3)   => ituStat.gtByteReAlig,
         sts32array_o(0+4) => ituStat.CrcCnt,
         sts32array_o(1+4) => ituStat.LenCnt,
         sts32array_o(2+4) => ituStat.DropCnt,
         sts32array_o(3+4) => ituStat.tmstpStallCnt,
         sts32array_o(4+4) => ituStat.ToutCnt,
         sts32array_o(5+4) => ituStat.ddsParityCnt,
         link_o            => ituStat.link
      );
   ituStat.ddsLinkCnt <= (others => '0');
   --! @brief Combinational process
   --! @details Main module logic. Generates rin based on r and any module inputs
   comb : process (rst_i, clk_i, r, hpdStrobe, ituMsg) is
      variable v : RegType;
   begin
      -- Latch the current value
      v            := r;
      v.tmstpStall := '0';

      -- timestamp register
      if (hpdStrobe = '1') then
         v.timeStampReg := ituMsg.timeStamp;
         if ituMsg.timeStamp = r.timeStampReg then
            --! new timestamp is the same as the old timestamp => Stall
            v.tmstpStall := '1';
         end if;
      end if;

      -- Reset
      if (rst_i = '1') then
         v := REG_INIT_C;
      end if;

      -- Register the variable for next clock cycle
      rin <= v;
   end process comb;

   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk to create registers
   seq : process (clk_i) is
   begin
      if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
      end if;
   end process seq;

   --! Output assignment
   ituStat_o <= ituStat;
   ituMsg_o  <= ituMsg;

end structure;
---------------------------------------------------------------------------------------------------