---------------------------------------------------------------------------------------------------
--! @brief Cosylab Axi Stream Resize module
--! @details 
--!   Module is resizing Axi Stream to different configurations. TID, TDEST, TUSER bit widths are 
--!   configured through generics for both slave and master ports. With generics BYTES_MST_G and 
--!   BYTES_SLV_G number of bytes in slave and master ports is determined. BYTES_MST_G and BYTES_SLV_G
--!   must be multiples of each other for correct operation. Output of the downsizing is LSB byte first.
--!   Input to upsizing is LSB byte first.
--!
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date Feb 22 2019 created
--! @date Apr 09 2018 last modify
--!
--! @version 1.0
--!
--! @file CslAxisResize.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_arith.all;
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
---------------------------------------------------------------------------------------------------
entity CslAxisResize is
   generic (
      -- General Configurations
      TPD_G          : time      := 1 ns; 
      TID_WIDTH_G    : positive  := 8;    --! width of TID vector
      TDEST_WIDTH_G  : positive  := 8;    --! width of TDEST vector
      TUSER_WIDTH_G  : positive  := 8;    --! width of TUSER vector per byte
      BYTES_SLV_G    : positive  := 8;    --! nubmer of bytes in slave port
      BYTES_MST_G    : positive  := 8);   --! number of bytes in master port
   port (
      -- Clock and reset
      clk_i          : in  sl;                    --! clock signal bus
      rst_i          : in  sl;                    --! reset signal bus
  
      -- Slave Port
      sAxisMaster_i  : in  AxiStreamMasterType;   --! slave axis input signals
      sAxisSlave_o   : out AxiStreamSlaveType;    --! slave axis output signals
  
      -- Master Port
      mAxisMaster_o  : out AxiStreamMasterType;   --! master axis output signals
      mAxisSlave_i   : in  AxiStreamSlaveType     --! master axis input signals
   );
end CslAxisResize;
---------------------------------------------------------------------------------------------------
architecture rtl of CslAxisResize is

   --! internal constant
   constant COUNT_C : natural := ite(BYTES_SLV_G > BYTES_MST_G, BYTES_SLV_G / BYTES_MST_G, BYTES_MST_G / BYTES_SLV_G);

   --! Record containing all register elements
   type RegType is record
      cnt         : natural;
      outMaster   : AxiStreamMasterType;
      inSlave     : AxiStreamSlaveType;
   end record RegType;

   --! Initial and reset values for all register elements 
   constant REG_INIT_C : RegType := (
      cnt         => 0,
      outMaster   => AXI_STREAM_MASTER_INIT_C,
      inSlave     => AXI_STREAM_SLAVE_INIT_C
   );

   signal nRst : sl;

   --! Output of registers
   signal r   : RegType := REG_INIT_C;
   
   --! Combinatorial input to registers
   signal rin : RegType;
---------------------------------------------------------------------------------------------------
begin

   nRst <= not rst_i;

   -- Make sure data widths are appropriate. 
   assert ((BYTES_SLV_G >= BYTES_MST_G and BYTES_SLV_G mod BYTES_MST_G = 0) or
           (BYTES_MST_G >= BYTES_SLV_G and BYTES_MST_G mod BYTES_SLV_G = 0))
      report "Data widths must be even number multiples of each other" severity failure;
   
   --!generated reconnection slaves <=> master ports as resize is not needed 
   GEN_EQUAL: if BYTES_SLV_G = BYTES_MST_G generate

      --! Disable outputs when in reset (synchronous)
      reset_eq : process (clk_i, nRst, sAxisMaster_i, mAxisSlave_i) is
      begin
         if (rising_edge(clk_i)) then
            if (nRst = '0') then -- in reset
               mAxisMaster_o     <= AXI_STREAM_MASTER_INIT_C after TPD_G;
               sAxisSlave_o      <= AXI_STREAM_SLAVE_INIT_C after TPD_G;
            else               -- not in reset
               mAxisMaster_o     <= sAxisMaster_i after TPD_G;
               sAxisSlave_o      <= mAxisSlave_i after TPD_G;
            end if;
         end if;
      end process reset_eq;

   end generate GEN_EQUAL;
   
   --!generated resizing => master port is bigger than slave port 
   GEN_EXPAND: if BYTES_SLV_G < BYTES_MST_G generate
      comb : process (mAxisSlave_i, sAxisMaster_i, r, rst_i) is
         variable v       : RegType;
      begin
         v                          := r;         
         v.inSlave.tReady           := '0';
         
         -- clear valid when slave tReady is detected
         if mAxisSlave_i.tReady = '1' then
            v.outMaster.tValid      := '0';
         end if;
         
         -- Pipeline advance when the data is not prepared 
         if v.outMaster.tValid = '0' then
            --! when increasing port => slave is always ready
            v.inSlave.tReady        := '1';
            
            -- init when count = 0
            if (r.cnt = 0) then
               v.outMaster          := AXI_STREAM_MASTER_INIT_C;
               v.outMaster.tKeep    := (others=>'0');
               v.outMaster.tStrb    := (others=>'0');
            end if;
            
            --! parse slave signals to master signals
            v.outMaster.tId(TID_WIDTH_G-1 downto 0)      := sAxisMaster_i.tId(TID_WIDTH_G-1 downto 0);
            v.outMaster.tDest(TDEST_WIDTH_G-1 downto 0)  := sAxisMaster_i.tDest(TDEST_WIDTH_G-1 downto 0);
            v.outMaster.tLast                            := sAxisMaster_i.tLast;
            
            v.outMaster.tData((BYTES_SLV_G*8*(r.cnt+1)) - 1 downto (BYTES_SLV_G*8*r.cnt))  
                                                         := sAxisMaster_i.tData((BYTES_SLV_G*8)-1 downto 0);
                                                         
            v.outMaster.tUser((BYTES_SLV_G*TUSER_WIDTH_G*(r.cnt+1)) - 1 downto (BYTES_SLV_G*TUSER_WIDTH_G*r.cnt))  
                                                         := sAxisMaster_i.tUser((BYTES_SLV_G*TUSER_WIDTH_G)-1 downto 0);
                                                         
            v.outMaster.tStrb((BYTES_SLV_G*(r.cnt+1)) - 1 downto (BYTES_SLV_G*r.cnt))          
                                                         := sAxisMaster_i.tStrb(BYTES_SLV_G-1 downto 0);
                                                         
            v.outMaster.tKeep((BYTES_SLV_G*(r.cnt+1)) - 1 downto (BYTES_SLV_G*r.cnt))          
                                                         := sAxisMaster_i.tKeep(BYTES_SLV_G-1 downto 0);
            -- Determine if we move data
            if sAxisMaster_i.tValid = '1' then
               if r.cnt = (COUNT_C-1) or sAxisMaster_i.tLast = '1' then
                  --! slave signals were parsed to master signal or tLast from slave was asserted
                  v.outMaster.tValid   := '1';        --! master signals are valid
                  v.cnt                := 0;          --! reset counter
               else
                  v.cnt                := r.cnt + 1;  --! continue with parsing
               end if;
            end if;
         end if;

         -- Reset
         if (rst_i = '1') then
            v := REG_INIT_C;
         end if;
         
         rin            <= v;
         
         --! Output assignment
         sAxisSlave_o   <= v.inSlave;
         mAxisMaster_o  <= r.outMaster;
      end process comb;
   end generate GEN_EXPAND;
   
   --!generated resizing => master port is smaller than slave port 
   GEN_NARROW: if BYTES_SLV_G > BYTES_MST_G generate
      comb : process (mAxisSlave_i, sAxisMaster_i, r, rst_i) is
         variable v       : RegType;
         variable byteCnt : natural; -- Number of valid bytes in incoming bus
         variable bytes   : natural; -- byte version of counter
      begin
         v                          := r;
         --! number of bytes parsed from slave port
         bytes                      := (r.cnt+1) * BYTES_MST_G;   
         --! number of bytes with tKeep flag from slave port
         byteCnt                    := conv_integer(sAxisMaster_i.tKeep(BYTES_SLV_G-1 downto 0));
         
         -- Init ready
         v.inSlave.tReady           := '0';
            
         -- clear valid when slave tReady is detected
         if mAxisSlave_i.tReady = '1' then
            v.outMaster.tValid      := '0';
         end if;

         -- Pipeline advance when the data is not prepared 
         if v.outMaster.tValid = '0' then
            --! init out master port signals
            v.outMaster             := AXI_STREAM_MASTER_INIT_C;
            v.outMaster.tKeep       := (others=>'0');
            v.outMaster.tStrb       := (others=>'0');
            
            --! parse slave signals to master signals
            v.outMaster.tData((BYTES_MST_G*8)-1 downto 0) := sAxisMaster_i.tData((BYTES_MST_G*8*(r.cnt+1)) - 1 downto (BYTES_MST_G*8*r.cnt));
            v.outMaster.tUser((BYTES_MST_G*TUSER_WIDTH_G)-1 downto 0) 
                                    := sAxisMaster_i.tUser((BYTES_MST_G*TUSER_WIDTH_G*(r.cnt+1)) - 1 downto (BYTES_MST_G*TUSER_WIDTH_G*r.cnt));
            v.outMaster.tStrb(BYTES_MST_G-1 downto 0)     := sAxisMaster_i.tStrb((BYTES_MST_G*(r.cnt+1)) - 1 downto (BYTES_MST_G*r.cnt));
            v.outMaster.tKeep(BYTES_MST_G-1 downto 0)     := sAxisMaster_i.tKeep((BYTES_MST_G*(r.cnt+1)) - 1 downto (BYTES_MST_G*r.cnt));

            v.outMaster.tId         := sAxisMaster_i.tId;
            v.outMaster.tDest       := sAxisMaster_i.tDest;
            
            if sAxisMaster_i.tValid = '1' then 
               if (r.cnt = (COUNT_C-1)) or ((bytes >= byteCnt) and (sAxisMaster_i.tLast = '1')) then
                  --! all bytes with tKeep from slave port were parsed to master port
                  v.cnt             := 0;                   --! reset counter
                  v.inSlave.tReady  := '1';                 --! ready to parse new slave data
                  v.outMaster.tLast := sAxisMaster_i.tLast; 
               else
                  v.cnt             := r.cnt + 1;           --! continue with parsing
                  v.inSlave.tReady  := '0';
                  v.outMaster.tLast := '0';
               end if;
            end if;

            -- Drop transfers with no tKeep bits set, except on tLast
            v.outMaster.tValid := sAxisMaster_i.tValid and (uOr(v.outMaster.tKeep(COUNT_C-1 downto 0)) or v.outMaster.tLast);
         end if;

         -- Reset
         if (rst_i = '1') then
            v := REG_INIT_C;
         end if;
         
         rin                     <= v;
        
         --! Output assignment
         sAxisSlave_o            <= v.inSlave;
         mAxisMaster_o           <= r.outMaster;
      end process comb;
   end generate GEN_NARROW;
      
   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk to create registers
   seq : process (clk_i) is
   begin
      if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
      end if;
   end process seq;

end rtl;
---------------------------------------------------------------------------------------------------