---------------------------------------------------------------------------------------------------
--! @brief  CslAxilRead.vhd testbench
--! @details  Testbench employs BRAM with axiLite interface which is initialized with values from .coe file (coefs.coe). Copy of that
--!           file (coefsTb.coe) is parsed by this tesbench. The tesbench performs NUM_OF_READS_C reads from RAM 
--!           and compares read data (data_o) with data parsed from file. If data matches, transaction was successfull. 
--!           Axi Lite interface is connected to Xilinx axi protocol checker to ensure it works according to the axi specifications.
--!
--! @author Jost Reberc(jost.reebrc@cosylab.com)
--!
--! @date March 15 2019 created
--!
--! @version 
--!
--! @file CslAxilReadTb.vhd
---------------------------------------------------------------------------------------------------
library ieee; 
use ieee.std_logic_1164.all; 
use ieee.numeric_std.all; 
use ieee.std_logic_unsigned.all; 
use work.CslStdRtlPkg.all; 
use work.CslAxiPkg.all; 
use work.CslTbPkg.all; 
use work.CslPrintfPkg.all; 
use ieee.std_logic_textio.all; 
use std.textio.all; 
---------------------------------------------------------------------------------------------------
entity CslAxilReadTb is
end CslAxilReadTb; 
---------------------------------------------------------------------------------------------------
architecture behavior of CslAxilReadTb is 
   
   --Inputs
   signal clk_i           : sl := '0'; --! internal clock signal bus
   signal rst_i           : sl := '0'; --! internal reset signal bus
   signal valid_i         : sl; 
   signal addr_i          : slv(31 downto 0); 
   signal axilReadSlave_i : AxiLiteReadSlaveType; 
   
   --Outputs
   signal valid_o          : sl; 
   signal data_o           : slv(31 downto 0); 
   signal ready_o          : sl; 
   signal err_o            : sl; 
   signal axilReadMaster_o : AxiLiteReadMasterType; 
   
   -- other signals
   file fileVectors : text; 
   
   -- Clock period definitions 
   constant T_C            : time             := 10.0 ns; --! Clock period constant
   constant TPD_C          : time             := 1.0 ns;  --! Propagation delay
   constant NUM_OF_READS_C : integer          := 10; 
   constant ONE_VECT_C     : slv(31 downto 0) := (others => '1'); 
   constant ZERO_VECT_C    : slv(31 downto 0) := (others => '0'); 
---------------------------------------------------------------------------------------------------
begin
   
   --! Instantiate the Unit Under Test (UUT)
   uut : entity work.CslAxilRead
      generic map(
         --! Propagation delay
         TPD_G       => TPD_C,
         VALID_RED_G => true
      )
      port map( 
         clk_i            => clk_i,
         rst_i            => rst_i,            --: in  sl;
         valid_i          => valid_i,          --: in  sl;
         valid_o          => valid_o,          --: out sl;
         data_o           => data_o,           --: out slv(31 downto 0);
         addr_i           => addr_i,           --: in  slv(31 downto 0);
         ready_o          => ready_o,          --: out sl;
         err_o            => err_o,            --: out sl;
         axilReadMaster_o => axilReadMaster_o, --: out AxiLiteReadMasterType;
         axilReadSlave_i  => axilReadSlave_i   --: in  AxiLiteReadSlaveType
      ); 
   
   U_axiSlave : entity work.axiLiteRam
      port map (
         s_aclk        => clk_i,
         s_aresetn     => not rst_i,
         s_axi_awaddr  => ZERO_VECT_C,              --: IN STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_awvalid => ZERO_VECT_C(0),           --: IN STD_LOGIC;
         s_axi_awready => open,                     --: OUT STD_LOGIC;
         s_axi_wdata   => ZERO_VECT_C,              --: IN STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_wstrb   => ZERO_VECT_C(3 downto 0),  --: IN STD_LOGIC_VECTOR(3 DOWNTO 0);
         s_axi_wvalid  => ZERO_VECT_C(0),           --: IN STD_LOGIC;
         s_axi_wready  => open,                     --: OUT STD_LOGIC;
         s_axi_bresp   => open,                     --: OUT STD_LOGIC_VECTOR(1 DOWNTO 0);
         s_axi_bvalid  => open,                     -- : OUT STD_LOGIC;
         s_axi_bready  => ZERO_VECT_C(0),           --: IN STD_LOGIC;
         s_axi_araddr  => axilReadMaster_o.araddr,  --: IN STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_arvalid => axilReadMaster_o.arvalid, --: IN STD_LOGIC;
         s_axi_arready => axilReadSlave_i.arready,  --: OUT STD_LOGIC;
         s_axi_rdata   => axilReadSlave_i.rdata,    --: OUT STD_LOGIC_VECTOR(31 DOWNTO 0);
         s_axi_rresp   => axilReadSlave_i.rresp,    -- : OUT STD_LOGIC_VECTOR(1 DOWNTO 0);
         s_axi_rvalid  => axilReadSlave_i.rvalid,   --: OUT STD_LOGIC;
         s_axi_rready  => axilReadMaster_o.rready   --: IN STD_LOGIC
      ); 
   
   U_axiLiteReadCheck : entity work.axiLiteReadChecker
      port map (
         pc_status      => open,
         pc_asserted    => open,
         aclk           => clk_i,
         aresetn        => not rst_i,
         pc_axi_araddr  => axilReadMaster_o.araddr,
         pc_axi_arprot  => ZERO_VECT_C(2 downto 0),
         pc_axi_arvalid => axilReadMaster_o.arvalid,
         pc_axi_arready => axilReadSlave_i.arready,
         pc_axi_rdata   => axilReadSlave_i.rdata,
         pc_axi_rresp   => axilReadSlave_i.rresp,
         pc_axi_rvalid  => axilReadSlave_i.rvalid,
         pc_axi_rready  => axilReadMaster_o.rready
      ); 
   
   --! @brief Clock procces 
   --! @details Clock signal generator
   --! @param[in]  T_C
   --! @param[out] clk_i
   p_SyncClkGen : process
   begin
      clk_i <= '1'; 
      wait for T_C/2; 
      clk_i <= '0'; 
      wait for T_C/2; 
   end process; 
   
   --! @brief 
   --! @details 
   --! @param[in] 
   --! @param[out]
   p_Sim : process
      variable vIline_v  : line; 
      variable ramVect_v : std_logic_vector(31 downto 0); 
   begin
      
      file_open(fileVectors, "coefsTb.coe", read_mode); 
      readline(fileVectors, vIline_v); --read info line
      readline(fileVectors, vIline_v); --read info line
                                       -- input signals
      valid_i <= '0'; 
      rst_i   <= '0'; 
      wait for T_C * 10; 
      
      -- reset
      rst_i <= '1'; 
      wait for T_C * 10; 
      rst_i <= '0'; 
      wait for T_C * 10; 
      
      -- simulate--
      addr_i <= (others => '0'); --first address
      wait for T_C * 10; 
      
      for I in 0 to NUM_OF_READS_C-1 loop --make NUM_OF_READS_C reads
         
         valid_i <= '1'; 
         wait for T_C * 1; 
         if(ready_o = '0')then --check if component is ready
            wait until ready_o = '1'; 
         end if; 
         valid_i <= '0'; 
         addr_i  <= addr_i+4;      --addres increase
         wait until valid_o = '1'; --wait data
         if(err_o = '0')then
            if(not endfile(fileVectors))then
               readline(fileVectors, vIline_v); 
               hread(vIline_v, ramVect_v); 
               tbCheckSlv(data_o,ramVect_v,"Data does not match! ",T_C); 
            else
               printf(":: Data file ended \@"); 
            end if; 
         else
            printf(":: Error detected!  \@"); 
            assert false severity failure; 
         end if; 
      end loop; 
      
      wait for T_C * 10; 
      
      file_close(fileVectors); 
      
      -- Stop simulation
      assert false report "SIMULATION COMPLEATED" severity failure; 
      
   end process; 
end; 
---------------------------------------------------------------------------------------------------
