---------------------------------------------------------------------------------------------------
--! @brief  Asynchronous FIFO module
--! @details The module write to FIFO buffer from one clock domain and the data values are read 
--! from the same FIFO buffer from different clock domain where the two clock domains are 
--! asynchronous to each other. With generic AWIDTH_G address width is determined and with generic
--! DWIDTH_G data width is determined. Generic PIPELINE_G determines clock delays on FIFO read data
--! output. With generic BRAM_G RAM type is determined, 
--! BRAM_G=false => LUT RAM, BRAM_G=true => BRAM
--! 
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date January 17 2018 created
--! @date March 29 2018 last modify
--!
--! @version v1.0
--!
--! @par Modifications:
--! jkokalj, January 17 2018: Created
--! jkokalj, January 23 2018: added Simple Dual Port RAM
--! jkokalj, March 28 2018: design iteration
--!
--! @file CslAsyncFIFO.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslStdRtlPkg.all;

--! @brief  Asynchronous FIFO module
--! @details The module write to FIFO buffer from one clock domain and the data values are read 
--! from the same FIFO buffer from different clock domain where the two clock domains are 
--! asynchronous to each other: - full and empty flag are delayed for two clock cylces because of 
--! pointer synchornisation to other clock domain
---------------------------------------------------------------------------------------------------
entity CslAsyncFIFO is
   generic(
      TPD_G          : time      := 1 ns;
      AWIDTH_G       : positive  := 4;     --! Address width
      DWIDTH_G       : positive  := 8;     --! Data width
      PIPELINE_G     : natural   := 1;     --! delay in clock cycles of output data
      WRITE_FIRST_G  : boolean   := false;
      BRAM_G         : boolean   := true   --! false => LUT RAM, true => BRAM
   );
   Port ( 
      wclk_i         : in  sl;                           --!write local clock bus
      wrst_i         : in  sl;                           --!write reset signal bus
      winc_i         : in  sl;                           --!write pointer increment
      wdata_i        : in  slv (DWIDTH_G - 1 downto 0);  --! write data signal vector bus
      wfull_o        : out sl; --!FIFO is full => WR pointer wraped one more time than WR pointer
      
      rclk_i         : in  sl;                           --!read local clock bus
      rrst_i         : in  sl;                           --!read reset signal bus 
      rinc_i         : in  sl;                           --!read pointer increment
      rdata_o        : out slv (DWIDTH_G - 1 downto 0);  --! read data signal vector bus
      rempty_o       : out sl;  --!FIFO is empty => RD pointer caches WR pointer
      
      flush_i        : in  sl    := '0';                 --! empty FIFO and reset all counters
      rcnt_o         : out slv(AWIDTH_G downto 0);       --! fifo fullness counter
      unf_o          : out sl;                           --! underflow flag signal
      ovf_o          : out sl                            --! overflow flag signal
      );
end CslAsyncFIFO;

architecture rtl of CslAsyncFIFO is

   signal rptr  : slv (AWIDTH_G downto 0);      --! read pointer in grey output
   signal wptr  : slv (AWIDTH_G downto 0);      --! write pointer in grey output
   signal raddr : slv (AWIDTH_G - 1 downto 0);  --! read address signal vector bus
   signal waddr : slv (AWIDTH_G - 1 downto 0);  --! write address signal vector bus
   signal wfull : sl;                           --! FIFO is full signal
   signal wea   : sl;                           --! FIFO RAM write enable signal
   signal wrst  : sl;                           --! write part of fifo reset
   signal rrst  : sl;                           --! read part of fifo reset
   
   signal rsync_wptr : slv (AWIDTH_G downto 0); --! synchronised WR pointer to RD domain
   signal wsync_rptr : slv (AWIDTH_G downto 0); --! synchronised RD pointer to WR domain
   
begin
   
   --! reset fifo on dedicated rst signal or on flush signal
   wrst     <= wrst_i or flush_i;
   rrst     <= rrst_i or flush_i;
   
   --! read pointer synchronised to write clock domain
   U_r2w:entity work.CslSyncVec 
      generic map(
         TPD_G  => TPD_G,
         WIDTH_G => AWIDTH_G+1,	--! width of data at intput/output bus
         DEPTH_G => 2	         --! number of FF's
      )
      Port map( 
         --global input signals
         clk_i => wclk_i,     --! input clock bus
         rst_i => wrst,       --! Input reset bus
         sig_i => rptr,       --! Input vector signal bus
         --global output signals
         sig_o => wsync_rptr  --! Output vector signal bus
      );
   
   --! write pointer synchronised to read clock domain
   U_w2r:entity work.CslSyncVec
      generic map(
         TPD_G   => TPD_G,
         WIDTH_G => AWIDTH_G+1,	--! width of data at intput/output bus
         DEPTH_G => 2	         --! number of FF's
      )
      Port map( 
         --global input signals
         clk_i => rclk_i,     --! input clock bus
         rst_i => rrst,       --! Input reset bus
         sig_i => wptr,       --! Input vector signal bus
         --global output signals
         sig_o => rsync_wptr  --! Output vector signal bus
      );
   
   --! Read pointer module
   U_rptr_empty:entity work.CslRptrEmpty
      generic map(
         TPD_G  => TPD_G,
         AWIDTH_G => AWIDTH_G          --! Address width
      )
      Port map ( 
         rclk_i        => rclk_i,      --! read local clock bus
         rrst_i        => rrst,        --! read reset signal bus 
         rinc_i        => rinc_i,      --! read pointer increment 
         rsync_wptr_i  => rsync_wptr,  --! synchronised write pointer to read domain
         rptr_o        => rptr,        --! read pointer in grey output  
         raddr_o       => raddr,       --! read address signal bus
         rempty_o      => rempty_o,    --! signal FIFO is empty => RD pointer caches WR pointer
         rcnt_o        => rcnt_o,      --! fifo fullness counter
         unf_o         => unf_o        --! underflow flag signal
      );
      
   --! Write pointer module   
   U_wptr_full:entity work.CslWptrFull
      generic map(
         TPD_G  => TPD_G,
         AWIDTH_G => AWIDTH_G          --! Address width
      )
      Port map ( 
         wclk_i        => wclk_i,      --! write local clock bus
         wrst_i        => wrst,        --! write reset signal bus
         winc_i        => winc_i,      --! write pointer increment 
         wsync_rptr_i  => wsync_rptr,  --! synchronised read pointer to write domain
         wptr_o        => wptr,        --! write pointer in grey output
         waddr_o       => waddr,       --! write address signal bus
         wfull_o       => wfull,       --! signal FIFO is full 
         ovf_o         => ovf_o        --! overflow flag signal
      );
             
   wea   <= winc_i and not(wfull);
   
   --! FIFO buffer/memory module - Simple Dual Port RAM
   U_FIFO_RAM:entity work.CslDpRAM
      generic map(
         TPD_G          => TPD_G,
         ADR_W_G        => AWIDTH_G,      --! Address width
         DAT_W_G        => DWIDTH_G,      --! Data width
         PIPELINE_G     => PIPELINE_G,    --! delay in clock cycles of output data
         WRITE_FIRST_G  => WRITE_FIRST_G,
         BRAM_G         => BRAM_G
      )
      Port map ( 
         clka     => wclk_i,  --! local A clock bus
         rsta     => wrst,    --! no signal B vector data out bus on Simple Dual Port RAM
         wea      => wea,     --! signal A write enable bus
         addra    => waddr,   --! signal A vector address bus
         dina     => wdata_i, --! signal A vector data in bus
         
         clkb     => rclk_i,  --! local B clock bus 
         rstb     => rrst,    --! signal B reset bus
         addrb    => raddr,   --! signal B vector address bus
         doutb    => rdata_o  --! signal B vector data out bus
      );
      
   -- drive output
   wfull_o <= wfull;
   
end;
---------------------------------------------------------------------------------------------------