---------------------------------------------------------------------------------------------------
--! @brief  Asynchronous FIFO module adapter for stream
--! @details The module has an internal distributed ram fifo that it tries to fill up 
--! from the bram FIFO it is attached to. Having the data available in the distributed
--! ram, it is able to deliver axi stream
--! DWIDTH_G data width is defined. Generic PIPELINE_G determines clock delays on FIFO read data
--! output. 
--! 
--! @author Blaz Kelbl, Cosylab (blaz.kelbl@cosylab.com)
--!
--! @date February 23 2018 created
--! @date 
--!
--! @version v1.0
--!
--! @par Modifications:
--! bkelbl, February 23 2019: Created
--! @file CslStreamFifoReadAdapter.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslStdRtlPkg.all;

--! @brief  Asynchronous FIFO module adapter for stream
--! @details The module has an internal distributed ram fifo that it tries to fill up 
--! from the bram FIFO it is attached to. Having the data available in the distributed
--! ram, it is able to deliver axi stream
--! DWIDTH_G data width is defined. Generic PIPELINE_G determines clock delays on FIFO read data
--! output. 
---------------------------------------------------------------------------------------------------
entity CslStreamFifoReadAdapter is
   generic(
      TPD_G          : time     := 1 ns;
      DWIDTH_G       : positive := 8;     --! Data width
      PIPELINE_G     : natural  := 2      --! delay in clock cycles of input data
   );
   Port ( 
      clk_i 		: in  sl; --!clock
      rst_i 		: in  sl; --!reset 
      
      rinc_o 		: out sl; --!pop the BRAM fifo
      rdata_i 		: in  slv (DWIDTH_G - 1 downto 0); --! read data signal vector bus
      rempty_i    : in  sl;
      
      ready_i     : in  sl;                        --! AXI stream compatible ready 
      valid_o     : out sl;                        --! AXI stream compatible valid
      data_o      : out slv(DWIDTH_G - 1 downto 0) --! stream data

   ); --!FIFO is empty => RD pointer caches WR pointer
end CslStreamFifoReadAdapter;

architecture rtl of CslStreamFifoReadAdapter is

   constant AWIDTH_C : positive := bitSize(PIPELINE_G);
   
   type reg_type is record
      rptr        : slv (AWIDTH_C downto 0); --! read pointer 
      wptrActual  : slv (AWIDTH_C downto 0); --! Actual write pointer (ram address, emptiness check)
      wptrVirtual : slv (AWIDTH_C downto 0); --! Virtual write pointer (fullness check)
      vld         : slv(PIPELINE_G - 1 downto 0);
   end record reg_type;
   
   signal v,r,rin : reg_type;
	
begin
   
   --! If there is no pipeline convert empty to valid, ready to read
   GEN_NO_PIPELINE : if PIPELINE_G = 0 generate
      rinc_o   <= ready_i and not rempty_i;
      valid_o  <= not rempty_i;
      data_o   <= rdata_i;
   end generate;

   
   --! The logic is only required when there is pipeline in the pipelined FIFO
   GEN_PIPELINE : if PIPELINE_G > 0 generate
   
      comb : process(rst_i, rempty_i, ready_i, r) -- combinational process
         variable v : reg_type;
         variable empty_v : sl;
         variable full_v  : sl;
         variable rinc_v  : sl;
      begin
         v := r; -- default assignment
         
         -- emptiness is checked versus actual write pointer value, that is, write pointer value that is delayed by the ram pipeline
         -- because the valid_o is the function of emptiness
         empty_v := '0';
         if ( r.wptrActual(AWIDTH_C - 1 downto 0) = r.rptr(AWIDTH_C - 1 downto 0) ) then
            empty_v := r.wptrActual(AWIDTH_C) xnor r.rptr(AWIDTH_C);
         end if;
         
         -- fullness is checked versus virtual write pointer value, that is, write pointer as if in a nonpipelined fifo
         -- because the reading the fifo must be only done when there will be space in the fifo after all the read data is read (and virtual value updated in advance)
         full_v := '0';
         if ( r.wptrVirtual(AWIDTH_C - 1 downto 0) = r.rptr(AWIDTH_C - 1 downto 0) ) then
            full_v := r.wptrVirtual(AWIDTH_C) xor r.rptr(AWIDTH_C);
         end if;
         -- read async fifo is there is virtual space in the output fifo
         rinc_v := not rempty_i and not full_v;   
         
         -- pipeline read signal to compensate for the ram delay
   
         v.vld := rinc_v & r.vld(PIPELINE_G - 1 downto 1);
         
         if (rinc_v = '1') then
            v.wptrVirtual := slv(unsigned(v.wptrVirtual) + to_unsigned(1,AWIDTH_C+1));
         end if;
         
         -- increment actual write pointer when the data is actually available
         if (r.vld(0) = '1') then
            v.wptrActual := slv(unsigned(v.wptrActual) + to_unsigned(1,AWIDTH_C+1));
         end if;
         
         -- increment read pointer when ready in is applied to nonempty output fifo
         if (ready_i = '1' and empty_v = '0') then
            v.rptr := slv(unsigned(r.rptr) + to_unsigned(1,AWIDTH_C+1));
         end if;
         
         if (rst_i = '1') then
            v.vld := (others => '0');
            v.wptrActual := (others => '0');
            v.wptrVirtual := (others => '0');
            v.rptr := (others => '0');
         end if;
         
         rin <= v;
         
         -- assign outputs (data directly from memory)
         valid_o <= not empty_v;
         rinc_o   <= rinc_v;
         
      end process comb;
   
      seq : process(clk_i) -- sequential process
      begin
         if rising_edge(clk_i) then 
            r <= rin after TPD_G; 
         end if;
      end process seq;   
   
      u_Mem :  entity work.CslDpRAM 
         generic map (
            TPD_G          => TPD_G    ,     
            ADR_W_G        => AWIDTH_C ,  
            DAT_W_G        => DWIDTH_G ,  
            PIPELINE_G     => 0        ,  
            WRITE_FIRST_G  => false    ,  
            BRAM_G         => false             
         )
         port map( 
            clka           => clk_i    , 
            rsta           => rst_i    ,
            wea            => r.vld(0) ,
            addra          => r.wptrActual(AWIDTH_C - 1 downto 0) ,
            dina           => rdata_i  ,
            douta          => open     ,
            
            clkb           => clk_i    ,
            rstb           => rst_i    ,
            addrb          => r.rptr(AWIDTH_C - 1 downto 0)       ,
            doutb          => data_o   
         );

   end generate;
   
end architecture rtl;
---------------------------------------------------------------------------------------------------