---------------------------------------------------------------------------------------------------
--! @brief  Write pointer module
--! @details The module generates write pointer in grey code. With synchronised read pointer to 
--! write clock domain asserts full signal if write pointer wrap one more time than read pointer. 
--! Lower AWIDTH_G -1 bits from write pointer are for FIFO memory address write.
--! 
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date Jan 17 2018 created
--! @date Feb 22 2018 last modify
--!
--! @version v1.0
--!
--! @par Modifications:
--! jkokalj, Jan 17 2018: Created
--! jkokalj, Feb 22 2018: design iteration
--!
--! @file CslWptrFull.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslStdRtlPkg.all;

--! @brief  Write pointer module
--! @details The module generates write pointer in grey code. With synchronised read pointer to 
--! write clock domain asserts full signal if write pointer wrap one more time than read pointer. 
--! Lower AWIDTH_G -1 bits from write pointer are for fifo memory address write.
---------------------------------------------------------------------------------------------------
entity CslWptrFull is
   generic(
      TPD_G        : time   := 1 ns;
      AWIDTH_G     : positive:= 4                        --! Address width
   );
   Port ( 
      wclk_i         : in  sl;                           --! write local clock bus
      wrst_i         : in  sl;                           --! write reset signal bus
      
      winc_i         : in  sl;                           --! write pointer increment 
      wsync_rptr_i   : in  slv (AWIDTH_G downto 0);      --! synchronised read pointer to write domain
      
      wptr_o         : out slv (AWIDTH_G downto 0);      --! write pointer in grey output
      waddr_o        : out slv (AWIDTH_G -1 downto 0);   --! write address signal bus
      --! signal fifo is full => WR pointer wrap one more time than RD pointer
      wfull_o        : out sl;                           --! fifo is full
      ovf_o          : out sl);                          --! overflow flag
end CslWptrFull;
---------------------------------------------------------------------------------------------------
architecture rtl of CslWptrFull is

   --! Record containing all register elements
   type reg_type is record
      ovf         : sl;                            --! fifo overflow
      full        : sl;                            --! fifo full signal bus
      wbin        : unsigned(AWIDTH_G downto 0);   --! binary pointer signal bus
      wptr        : unsigned(AWIDTH_G downto 0);   --! grey pointer signal bus
   end record reg_type;
   
   --! Output of registers
   signal r: reg_type;
   
   --! Combinatorial input to registers
   signal rin: reg_type;
   
   --! Initial and reset values for all register elements
   constant REG_INIT_C : reg_type := (
      ovf         => '0',
      full        => '0',
      wbin        => (others => '0'),
      wptr        => (others => '0'));   
---------------------------------------------------------------------------------------------------     
begin

   --! @brief Combinational process
   --! @details Main module logic. Generates rin based on r and any module inputs
   --! @param[in]  wrst_i, winc_i, wsync_rptr_i, r
   --! @param[out] rin, wfull_o, waddr_o, wptr_o
   comb : process(wrst_i, winc_i, wsync_rptr_i, r) -- combinational process
   variable v : reg_type;
   begin
      v                 := r; -- default assignment
      v.ovf             := '0'; 
      
      if winc_i = '1' then
         if r.full = '0' then
            --! pointer increment
            v.wbin      := r.wbin + 1;
         else
            v.wbin      := r.wbin;
            v.ovf       := '1';        --! write pointer increment on full fifo => overflow
         end if;  
      end if;
      
      v.wptr            := grayEncode(v.wbin); --! function to convert binary => grey
      
      if sl(v.wptr(AWIDTH_G)) /= wsync_rptr_i(AWIDTH_G)      and
         sl(v.wptr(AWIDTH_G-1)) /= wsync_rptr_i(AWIDTH_G-1)  and 
         slv(v.wptr(AWIDTH_G-2 downto 0)) =wsync_rptr_i(AWIDTH_G-2 downto 0) then
         --WR pointer wraped one more time than RD pointer
         v.full         := '1';
      else
         v.full         := '0';
      end if;
         
      if wrst_i = '1' then -- reset condition
         v              := REG_INIT_C;
      end if;
      
      rin               <= v; -- drive register inputs
      -- drive outputs
      ovf_o             <= r.ovf;
      wfull_o           <= r.full;    
      waddr_o           <= slv(r.wbin(AWIDTH_G -1 downto 0));
      wptr_o            <= slv(r.wptr);
   end process comb;
 
   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk to create registers
   --! @param[in]  rin, clk_i
   --! @param[out] r 
   seq : process(wclk_i) -- sequential process
   begin
      if rising_edge(wclk_i) then 
         r <= rin after TPD_G; 
      end if;
   end process seq;

end rtl;
---------------------------------------------------------------------------------------------------