---------------------------------------------------------------------------------------------------
--! @brief  Stream FIFO adapter test bench
--! @details   The test bench simulates randomized control signals and verifies assertions
--! The testbench includes AsnycFIFO to properly model its behavior
--!
--! @author Blaz Kelbl, Cosylab (blaz.kelbl@cosylab.com)
--!
--! @date February 23 2019 created
--!
--! @version v1.0
--!
--! @par Modifications:
--! bkelbl, February 2019: Created
--!
--! @file CslStreamFifoReadAdapterTb.vhd
--------------------------------LIBRARY ieee;
library ieee;
USE ieee.std_logic_1164.ALL;
USE ieee.numeric_std.ALL;
use ieee.math_real.all;
 
ENTITY CslStreamFifoReadAdapterTb IS
   generic(
      TPD_G                : time     := 1 ns;
      DWIDTH_G             : positive := 8;     --! Data width
      PIPELINE_G           : natural  := 2;     --! delay in clock cycles of output data
      -- TB generics
      READY_PERCENT_G      : natural  := 50;
      DATARATE_PERCENT_G   : natural  := 50;
      SIM_LOOP_G           : natural  := 10000000
      
   );
END CslStreamFifoReadAdapterTb;
 
ARCHITECTURE behavior OF CslStreamFifoReadAdapterTb IS 
 
   --Inputs
   signal clk_i : std_logic := '0';
   signal rst_i : std_logic := '0';
   signal rdata_i : std_logic_vector(DWIDTH_G - 1 downto 0) := (others => '0');
   signal rempty_i : std_logic := '1';
   signal ready_i : std_logic := '0';
   
   -- internal simulation
   signal winc_i : std_logic := '0';
   signal wdata_i : std_logic_vector(7 downto 0) := (others => '0');
   signal wfull_o : std_logic;
   --Outputs
   signal rinc_o : std_logic;
   signal valid_o : std_logic;
   signal data_o : std_logic_vector(DWIDTH_G - 1 downto 0);

   -- Clock period definitions
   constant wclk_i_period : time := 10 ns;
   constant clk_i_period  : time := 10 ns;
 
   procedure genIntRand(
      variable  rand_num       : out integer;
      constant  range_of_rand  : in real;        --! the range of random values created will be 0 to +1000.
      variable  seed1          : inout positive; --! seed values for random generator
      variable  seed2          : inout positive               
   ) is
      variable rand  : real;            --! random real-number value in range 0 to 1.0  
   begin
      -- generate random number
      uniform(seed1, seed2, rand); 
      -- rescale, convert integer part 
      rand_num := integer(rand*range_of_rand);  
   end procedure genIntRand;

BEGIN
   uut_fifo:entity work.CslAsyncFIFO 
      generic map(
         TPD_G          => TPD_G,
         AWIDTH_G       => 5,
         DWIDTH_G       => 8,
         PIPELINE_G     => PIPELINE_G,
         WRITE_FIRST_G  => true,
         BRAM_G         => true
      )
      PORT MAP (
         wclk_i => clk_i,
         wrst_i => rst_i,
         winc_i => winc_i,
         wdata_i => wdata_i,
         wfull_o => wfull_o,
         rclk_i => clk_i,
         rrst_i => rst_i,
         rinc_i => rinc_o,
         rdata_o => rdata_i,
         rempty_o => rempty_i
      ); 
   -- Instantiate the Unit Under Test (UUT)
   uut: entity work.CslStreamFifoReadAdapter
   GENERIC MAP (
      TPD_G          => TPD_G      ,    
      DWIDTH_G       => DWIDTH_G   ,    
      PIPELINE_G     => PIPELINE_G     
   )
   PORT MAP (
          clk_i      => clk_i    ,
          rst_i      => rst_i    ,
          rinc_o     => rinc_o   ,
          rdata_i    => rdata_i  ,
          rempty_i   => rempty_i ,
          valid_o    => valid_o  ,
          ready_i    => ready_i  ,
          data_o     => data_o
        );

   -- Clock process definitions
   clk_i_process :process
   begin
      clk_i <= '0';
      wait for clk_i_period/2;
      clk_i <= '1';
      wait for clk_i_period/2;
   end process;
   
   rst_i_process :process
   begin
      rst_i <= '1';
      wait for clk_i_period * 10;
      rst_i <= '0';
      wait ;
   end process;

   
   -- Stimulus process
   stim_proc: process
      variable seed1: positive := 1; 
      variable seed2: positive := 1 ;        -- seed values for random generator
      variable rand: real;                   -- random real-number value in range 0 to 1.0  
      variable range_of_rand : real := 100.0;-- the range of random values created will be 0 to +1000.
      variable rand_num : integer;
      variable vdata : integer;
   begin      
      -- hold reset state for 100 ns.
      wait for 100 ns;   
      
      wait for clk_i_period*10;
      vdata := 0;
      -- insert stimulus here 
      for i in 0 to SIM_LOOP_G loop
         winc_i <= '0'; 
         if wfull_o = '0' then
            genIntRand(rand_num,100.0,seed1,seed2);
            if (rand_num < DATARATE_PERCENT_G) then
               winc_i <= '1'; 
               wdata_i <= std_logic_vector(to_unsigned(vdata,8));
               vdata := (vdata + 1) mod 256;
            end if;
         end if;
         wait for clk_i_period;
      end loop;
      wait;
   end process;

   -- Stimulus process
   stim_proc_axi: process
      variable seed1: positive := 2; 
      variable seed2: positive := 2 ;        -- seed values for random generator
      variable rand: real;                   -- random real-number value in range 0 to 1.0  
      variable range_of_rand : real := 100.0;-- the range of random values created will be 0 to +1000.
      variable rand_num : integer;
   begin      
      -- hold reset state for 100 ns.
      wait for 100 ns;   
      
      wait for clk_i_period*10;
      
      -- insert stimulus here 

      for i in 0 to SIM_LOOP_G loop
         genIntRand(rand_num,100.0,seed1,seed2);
         if (rand_num < READY_PERCENT_G) then
            ready_i <= '1'; wait for clk_i_period; 
         else
            ready_i <= '0'; wait for clk_i_period; 
         end if;
      end loop;
      ready_i <= '1'; wait for clk_i_period;
      ready_i <= '1'; wait for clk_i_period;
      ready_i <= '1'; wait for clk_i_period;
      ready_i <= '0'; wait for clk_i_period;
      ready_i <= '1'; wait for clk_i_period;
      ready_i <= '0'; wait for clk_i_period;
      ready_i <= '1'; wait for clk_i_period;
      ready_i <= '0'; wait for clk_i_period;
      ready_i <= '1'; wait for clk_i_period;
      wait;
   end process;

   p_verify : process(clk_i)
      variable test_x : std_logic_vector(DWIDTH_G - 1 downto 0);
      variable test_inc: boolean;
      variable data_q: std_logic_vector(DWIDTH_G - 1 downto 0);
      variable int_data_q : integer;
      variable int_data_o : integer;
      variable int_data_o_dec : integer;
   begin
      if rising_edge(clk_i) then
         if (valid_o = '1' and ready_i = '1') then
            test_x := data_o xor data_o;
            -- if old data is updated
            if ((data_q xor data_q) = "00000000") then
               int_data_q := to_integer(unsigned(data_q));
               int_data_o := to_integer(unsigned(data_o));
               if (int_data_o = 0) then
                  int_data_o_dec := 255;
               else 
                  int_data_o_dec := int_data_o - 1;
               end if;
               -- check if old data is one less than new data
               if int_data_o_dec /= int_data_q then
                  assert false report "data data violation: " & integer'image(int_data_o_dec) severity error;
--               else 
--                  assert false report "data_o: " & integer'image(int_data_o) & " | data_q: " & integer'image(int_data_q) severity note;
               end if;
            end if;
            -- check if new data is legal
            assert test_x = "00000000" report "data X violation" severity error;
            
            data_q := data_o;
            
         end if;
      end if;
   end process;


END;
