---------------------------------------------------------------------------------------------------
--! @brief  Shift register module test bench
--! @details   The test bench simulates the module with various input signals value,
--! lenghts and timing. Tested CslSyncFIFO vs CslSyncCntFIFO modules.
--!
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date Jan 11 2018 created
--! @date Feb 22 2019 last modify
--!
--! @version v1.0
--!
--! @par Modifications:
--! jkokalj, Jan 11 2018: Created
--! jkokalj, Feb 22 2018: design iteration
--!
--! @file CslShiftRegTb.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslStdRtlPkg.all;

--! @brief  Shift register module test bench
--! @details   The test bench simulates the module with various input signals value,
--! lenghts and timing. 
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
---------------------------------------------------------------------------------------------------
ENTITY CslSyncFIFOTb IS
   generic(
      TPD_G          : time     := 1 ns;
      AWIDTH_G       : positive := 4;     --! Address width
      DWIDTH_G       : positive := 8;     --! Data width
      PIPELINE_G     : natural  := 0;     --! delay in clock cycles of output data
      WRITE_FIRST_G  : boolean  := false;
      BRAM_G         : boolean  := false   --! false => LUT RAM, true => BRAM
   );
END CslSyncFIFOTb;
---------------------------------------------------------------------------------------------------
ARCHITECTURE behavioral of CslSyncFIFOTb is
   --Inputs
   signal clk_i : std_logic := '0';
   signal rst_i : std_logic := '0';
   
   signal winc_i : std_logic := '0';
   signal wdata_i : std_logic_vector(7 downto 0) := (others => '0');
   signal rinc_i : std_logic := '0';

 	--Outputs
   signal wfull_o : std_logic;
   signal rdata_o : std_logic_vector(7 downto 0);
   signal rempty_o : std_logic;
 
   constant clk_period_C : time := 10 ns;
---------------------------------------------------------------------------------------------------
BEGIN

   -- Instantiate the Unit Under Test (UUT)
   uut_CslSyncFIFO : entity work.CslSyncFIFO 
      generic map(
         TPD_G          => TPD_G,
         AWIDTH_G       => AWIDTH_G,
         DWIDTH_G       => DWIDTH_G,
         PIPELINE_G     => PIPELINE_G,
         WRITE_FIRST_G  => WRITE_FIRST_G,
         BRAM_G         => BRAM_G
      )
      PORT MAP (
         clk_i    => clk_i,
         rst_i    => rst_i,
         winc_i   => winc_i,
         wdata_i  => wdata_i,
         wfull_o  => wfull_o,
         rinc_i   => rinc_i,
         rdata_o  => rdata_o,
         rempty_o => rempty_o
      );

   -- Instantiate the Unit Under Test (UUT)
   uut_CslSyncCntFIFO : entity work.CslSyncCntFIFO
      generic map (
         TPD_G         => TPD_G,
         AWIDTH_G      => AWIDTH_G,
         DWIDTH_G      => DWIDTH_G,
         PIPELINE_G    => PIPELINE_G,
         WRITE_FIRST_G => WRITE_FIRST_G,
         BRAM_G        => BRAM_G
      )
      port map (
         clk_i    => clk_i,
         rst_i    => rst_i,
         winc_i   => winc_i,
         wdata_i  => wdata_i,
         wfull_o  => open,
         rinc_i   => rinc_i,
         rdata_o  => open,
         rcnt_o   => open,
         rempty_o => open
      );

   --! @brief Write Clock procces 
   --! @details Clock signal generator
   --! @param[in]  clk_i_period_C
   --! @param[out] clk_i
   p_ClkGen :process
   begin
		clk_i <= '0';
		wait for clk_period_C/2;
		clk_i <= '1';
		wait for clk_period_C/2;
   end process;
   
   --! @brief Reset procces 
   --! @details Reset signal generator
   --! @param[in]  clk_i_period_C
   --! @param[out] rst_i
   p_RstGen :process
   begin
		rst_i <= '1';
      --rrst_T <= '0';
		wait for clk_period_C*500;
		rst_i <= '0';
      --rrst_T <= '1';
		wait for clk_period_C*1500;
   end process; 
   
   --! @brief Simulation procces 
   --! @details Simulation with various signals value, length and timings
   --! @param[in] 
   --! @param[out] data_i, load_i 
   p_Sim: process
   begin		
      wait for 40*clk_period_C;
      rinc_i <= '1';
      wait for 2*clk_period_C ;	
      rinc_i <= '0';
      wait for 4*clk_period_C ;
      rinc_i <= '1';
      wait for 8*clk_period_C ;
      rinc_i <= '0';
      wait for 6*clk_period_C;	
      rinc_i <= '1';
      wait for 4*clk_period_C ;
      rinc_i <= '0';
      wait for 2*clk_period_C ;
      rinc_i <= '1';
      wait for 2*clk_period_C;	
      rinc_i <= '0';
      wait for 2*clk_period_C ;
      rinc_i <= '1';
      wait for 8*clk_period_C;	
      rinc_i <= '0';
   end process;
   
   --! @brief Simulation procces 
   --! @details Simulation with various signals value, length and timings
   --! @param[in] 
   --! @param[out] data_i, load_i 
   WRstim_proc: process
   begin		
      wait for 4*clk_period_C ;
      wdata_i <= "11110000";
      winc_i <= '1';
      wait for 2*clk_period_C ;	
      winc_i <= '0';
      wait for 2*clk_period_C ;
      wdata_i <= "11010000";      
      winc_i <= '1';
      wait for 2*clk_period_C ;
      wdata_i <= "11010110";      
      winc_i <= '0';
      wait for 8*clk_period_C ;	
      winc_i <= '1';
      wait for 2*clk_period_C ;	
      winc_i <= '0';
      wait for 4*clk_period_C ;
      wdata_i <= "11010100";
      winc_i <= '1';
      wait for 2*clk_period_C ;	
      winc_i <= '0';
      wait for 2*clk_period_C ;
      wdata_i <= "10101000";      
      winc_i <= '1';
      wait for 2*clk_period_C ;
      wdata_i <= "11011110";      
      winc_i <= '0';
      wait for 4*clk_period_C ;	
      winc_i <= '1';
      wait for 6*clk_period_C ;	
      winc_i <= '0';
   end process;

END;
---------------------------------------------------------------------------------------------------