---------------------------------------------------------------------------------------------------
--! @brief      Signal Vector Falling/Rising Edge Detection module
--! @details The module is detecting Rising and Falling edge of signal vector with no aditional 
--! clock cycles of delay. When direction generic DIR_G = FALSE, 
--! then rising edge detection is selected. 
--!
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date Januar 8 2018 created
--! @date Januar 8 2018 last modify
--!
--! @version v1.0
--!
--! @par Modifications:
--! jkokalj, Januar 8 2018: Created
--!
--! @file CslEdgeDetVec.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslStdRtlPkg.all;

--! @brief      Signal Vector Falling/Rising Edge Detection module
--! @details The module is detecting Rising and Falling edge of signal vector with no aditional 
--! clock cycles of delay. When direction generic DIR_G = FALSE, 
--! then rising edge detection is selected. 
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
---------------------------------------------------------------------------------------------------
entity CslEdgeDetVec is
   generic(
      TPD_G          : time     := 1 ns;
      -- set the direction
      --! FALSE(default) = rising edge detection, 
      --! TRUE = falling edge detection
      FALLING_EDGE_G : boolean := FALSE;
      WIDTH_G : natural:=8	--! data width at intput/output bus
   );
   port ( 
      --global input signals
      clk_i : in  sl;   --! input clock bus
      rst_i : in  sl;   --! Input reset bus
      sig_i : in  slv (WIDTH_G-1 downto 0);  --! Output signal vector bus
      --global output signals
      sig_o : out  slv (WIDTH_G-1 downto 0));--! Input signal vector bus
end CslEdgeDetVec;
---------------------------------------------------------------------------------------------------
architecture rtl of CslEdgeDetVec is
---------------------------------------------------------------------------------------------------
begin

   --!generated Edge detection vector component 
   gen_EdgeDetVec: for i in 0 to WIDTH_G - 1 generate 
      ux_EdgeDetBit: entity work.CslEdgeDet 
         generic map(
            TPD_G => TPD_G,
            FALLING_EDGE_G => FALLING_EDGE_G
            )
         port map (
            clk_i => clk_i,
            rst_i => rst_i,
            sig_i => sig_i(i),
            sig_o => sig_o(i)
         ); 
   end generate gen_EdgeDetVec;
   
end rtl;
---------------------------------------------------------------------------------------------------
