---------------------------------------------------------------------------------------------------
--! @brief  Signal Synchronizer Module
--! @details The module shifts input signal through a series of D flip flops.
--! With generic DEPTH_G the number of D flip flop is determined.
--!
--! Note: This module could be used for synchronisation from an unknown time domain.
--! 
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date January 8 2018 created
--! @date March 28 2018 last modify
--!
--! @version v1.0
--!
--! @par Modifications:
--! jkokalj, January 8 2018: Created
--! jkokalj, March 28 2018: design iteration
--!
--! @file CslSync.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslStdRtlPkg.all;

--! @brief  Signal Synchronizer Module
--! @details The module shifts input signal through a series of D flip flops.
--! With generic DEPTH_G the number of D flip flop is determined.
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
---------------------------------------------------------------------------------------------------
entity CslSync is
   generic(
      TPD_G    : time   := 1 ns;
		DEPTH_G  : natural:= 3      --!number of FF's
	);
   port(
      --global input signals
      clk_i : in sl; --! input clock bus
      rst_i : in sl; --! Input reset bus
      sig_i : in sl; --! Input signal bus
      --global output signals
      sig_o : out sl --! Output signal bus
   );
end CslSync;
---------------------------------------------------------------------------------------------------
architecture rtl of CslSync is
---------------------------------------------------------------------------------------------------
begin
   GEN_ASYNC : if DEPTH_G = 0 generate
      --! Asynchronised Output
      sig_o <= sig_i;
   end generate GEN_ASYNC;
   
   GEN_SYNC_ONE : if DEPTH_G = 1 generate
      signal dff : sl ;
   begin
      --! @brief Sync procces 
      --! @details D flip flop for synchronisation
      --! @param[in]  clk_i, rst_i,sig_i
      --! @param[out] dff 
      p_Sync : process(clk_i)
      begin								
         if rising_edge(clk_i) then
            if rst_i = '1' then
               --! Sync reset
               dff <= '0' after TPD_G; 
            else
               --! Input signal to D flip flop
               dff <= sig_i after TPD_G;
            end if;
         end if;
      end process;
      
      --! Synchronised Output
      sig_o<=dff;
   end generate GEN_SYNC_ONE;
   
   GEN_SYNC : if DEPTH_G > 1 generate
      signal dff : slv (DEPTH_G-1 downto 0); --! D flip flops
   begin
      --! @brief Sync procces 
      --! @details Shift register implemented with D flip flops
      --! @param[in]  clk_i, rst_i, DEPTH_G , sig_i
      --! @param[out] dff 
      p_Sync : process(clk_i)
      begin								
         if rising_edge(clk_i) then
            if rst_i = '1' then
               --! Sync reset
               dff <= (others => '0') after TPD_G; 
            else
               --! Input signal shifted
               dff <= dff(DEPTH_G-2 downto 0) & sig_i after TPD_G;
            end if;
         end if;
      end process;
      
      --! Synchronised Output
      sig_o<=dff(DEPTH_G-1);
   end generate GEN_SYNC;
end rtl;
---------------------------------------------------------------------------------------------------

