---------------------------------------------------------------------------------------------------
--! @brief  CRC16 Calculator
--! @details
--!   Calculates the 16bit CRC from 8-dit data with the following parameters.
--!   polynomial: x^16 + x^12 + x^5 + 1
--!   data width: 8
--!
--!   The calculation can be verified https://www.lammertbies.nl/comm/info/crc-calculation.html 
--!   look at CRC-CCITT (XModem) field
--!
--!   The CRC function nextCRC16_D8 was made by Easics on-line CRC generator
--!
--! @author Uros Laget, Cosylab (uros.legat@cosylab.com)
--!
--! @date Sep 20 2018 created
--! @date Sep 20 2018 last modify
--!
--! @version v0.1
--!
--!
--! @file Crc16D8.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;

use work.CslStdRtlPkg.all;


--! @brief  
--! @details 
---------------------------------------------------------------------------------------------------
entity Crc16D8 is
   generic (
      TPD_G          : time     := 1 ns);
   port (
      ------------
      -- Inputs --
      ------------

      --! Global clock
      clk_i: in std_logic;
      
      --! Global reset
      rst_i: in std_logic;
      
      --! Incoming message 8-bit data.
      data_i: in std_logic_vector(7 downto 0);
      
      --! Clear calculation.
      clear_i: in std_logic;
      
      --! When enabled calculate new CRC with the data on data_i.
      en_i: in std_logic;

      -------------
      -- Outputs --
      -------------
      
      --! Calculated 16-bit CRC.
      crc_o: out std_logic_vector(15 downto 0)
   );
end Crc16D8;


architecture rtl of Crc16D8 is
    
   -- CRC16 (Easics generator).
   -- polynomial: x^16 + x^12 + x^5 + 1
   -- data width: 8
   -- convention: the first serial bit is D[7]
   function nextCRC16_D8
      (Data: std_logic_vector(7 downto 0);
       crc:  std_logic_vector(15 downto 0))
      return std_logic_vector is

      variable d:      std_logic_vector(7 downto 0);
      variable c:      std_logic_vector(15 downto 0);
      variable newcrc: std_logic_vector(15 downto 0);

   begin
      d := Data;
      c := crc;

      newcrc(0) := d(4) xor d(0) xor c(8) xor c(12);
      newcrc(1) := d(5) xor d(1) xor c(9) xor c(13);
      newcrc(2) := d(6) xor d(2) xor c(10) xor c(14);
      newcrc(3) := d(7) xor d(3) xor c(11) xor c(15);
      newcrc(4) := d(4) xor c(12);
      newcrc(5) := d(5) xor d(4) xor d(0) xor c(8) xor c(12) xor c(13);
      newcrc(6) := d(6) xor d(5) xor d(1) xor c(9) xor c(13) xor c(14);
      newcrc(7) := d(7) xor d(6) xor d(2) xor c(10) xor c(14) xor c(15);
      newcrc(8) := d(7) xor d(3) xor c(0) xor c(11) xor c(15);
      newcrc(9) := d(4) xor c(1) xor c(12);
      newcrc(10) := d(5) xor c(2) xor c(13);
      newcrc(11) := d(6) xor c(3) xor c(14);
      newcrc(12) := d(7) xor d(4) xor d(0) xor c(4) xor c(8) xor c(12) xor c(15);
      newcrc(13) := d(5) xor d(1) xor c(5) xor c(9) xor c(13);
      newcrc(14) := d(6) xor d(2) xor c(6) xor c(10) xor c(14);
      newcrc(15) := d(7) xor d(3) xor c(7) xor c(11) xor c(15);
      return newcrc;
   end nextCRC16_D8; 
    
   --! Record containing all register elements
   type RegType is record
      crc : slv(15 downto 0);
      --

   end record RegType;

   --! Initial and reset values for all register elements   
   constant REG_INIT_C : RegType := (
      crc => (others => '0')
      ---
   );
   
   --! internal signals
   signal r     : RegType := REG_INIT_C;
   signal rin   : RegType;
---------------------------------------------------------------------------------------------------
begin
   --! @brief Combinational process
   --! @details Main module logic. Generates rin based on r and any module inputs
   comb : process (r, rst_i, data_i, clear_i, en_i) is
      variable v    : RegType;
   begin
      v := r;
      --! Calculate the CRC of the input data_i if enabled
      if(en_i = '1')then 
        v.crc := nextCRC16_D8(data_i, r.crc);
      end if;
      
      --! Clear the calculation before restarting the new payload
      if(clear_i = '1')then 
        v.crc := (others => '0');
      end if;
      
      --! Reset
      if (rst_i = '1') then
         v := REG_INIT_C;
      end if;

      rin <= v;
      
      --! Output assignment
      crc_o <= r.crc;
   -----------------------------------------------
   end process comb;
   
   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk to create registers
   seq : process (clk_i) is
   begin
      if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
      end if;
   end process seq;
------------------------------------------------------------------------------------
    
end rtl;