---------------------------------------------------------------------------------------------------
--! @brief  Fiber Link receiver module 
--! @details
--!   The receiver module accepts fiber messages and routes the data to PIPES_G number of AXI4-Stream pipes.
--!   The AXI4-Stream FIFO converts the 8-bit data to DATA_WIDTH.
--! 
--!   link_i -  The link_i indicates if the messages on the fiber link are valid. 
--!   If there is no link detection logic in the circuit this input can be tied to ‘1’.
--! 
--!   drop_o - output produces a one clock cycle pulse every time the message is dropped. 
--!   The message is dropped if the pipe number is out of allowed range, or the link is not established, 
--!   or the AXI4-Stream FIFO is full.
--!
--!   Two different clocks can be used to drive the CslAxisFifo for the purpose of clock domain crossing
--!   synchronization. This feature is enabled by setting TWO_CLOCKS_G to "true". If enabled, clk_i and 
--!   rst_i are used for the input side of the fifo and sysClk_i and sysRst_i are used for the output side.
--!   If this feature is disabled, clk_i and rst_i are used for driving both sides of the fifo.
--!   
--! @author Uros Laget, Cosylab (uros.legat@cosylab.com)
--!
--! @date Sep 27 2018 created
--! @date Aug 27 2019 last modify
--!
--! @version v0.1
--!
--! @file FiberLinkRx.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;
---------------------------------------------------------------------------------------------------
entity FiberLinkRx is
generic (
      TPD_G          : time     := 1 ns;               --! Simulation delta time offset
      DATA_WIDTH_G   : natural  := 4;                  --! Input axi stream width number of bytes (1,2,4,8,16)
      PIPES_G        : integer range 1 to 255 := 1;    --! Number of priority pipes (For HPD set to 1)        
      IDLE_G         : slv(8 downto 0) := '1' & x"BC"; --! Sent out at when IDLE_G
      SOF_CHAR_G     : slv(7 downto 0) := x"1C";       --! Start of frame character
      EOF_CHAR_G     : slv(7 downto 0) := x"FE";       --! End of frame character
      MAX_SIZE_G     : positive := 8;                  --! Maximum number of input axis data words 
      TWO_CLOCKS_G   : boolean  := false               --! Enable optional secondary clock and reset inputs
   );
   port (
      clk_i         : in  sl;
      rst_i         : in  sl;
      sysClk_i      : in  sl := '0';    --! optional secondary clock used for clock domain crossing
      sysRst_i      : in  sl := '0';    --! optional secondary reset used for clock domain crossing
      
      --! Fiber side interface connect directly to GT
      dataRx_i    : in slv(7 downto 0);
      charRx_i    : in sl;
      link_i      : in sl; 
      
      --! User side interface
      axisMaster_o : out AxiStreamMasterArray(PIPES_G-1 downto 0);
      axisSlave_i  : in  AxiStreamSlaveArray(PIPES_G-1 downto 0);      
      pktEmpty_o   : out slv(PIPES_G-1 downto 0);
      drop_o       : out sl
   );
end FiberLinkRx;
---------------------------------------------------------------------------------------------------
architecture rtl of FiberLinkRx is
   --! Calculation of the fifo parameters

   --! Max size in bytes
   constant MAX_BYTES_C  : positive := DATA_WIDTH_G*MAX_SIZE_G;    
   --! The fifo should have space for minimum of 4 (MAX_BYTES_C) packets 
   constant FIFO_ADDR_WIDTH_C  : positive := ite((bitSize(MAX_BYTES_C) < 4), 5, bitSize(MAX_BYTES_C)+2);
   --! The fifo pause threshold is set to indicate that there is space for one packet   
   constant FIFO_PAUSE_THRESH_C: positive := ((2**FIFO_ADDR_WIDTH_C) - MAX_BYTES_C);
      
   --! Word count define 32-bit width but only FIFO_ADDR_WIDTH_C will be used     
   signal fifoWrCnt  : Slv32Array(PIPES_G-1 downto 0):= (others => x"0000_0000");
   signal pktEmpty   : slv(PIPES_G-1 downto 0);         
   signal pktFull    : slv(PIPES_G-1 downto 0);
   
   --! Axi stream internal interconnect
   signal intAxisMaster_pi : AxiStreamMasterArray(PIPES_G-1 downto 0);
   signal intAxisSlave_po  : AxiStreamSlaveArray(PIPES_G-1 downto 0);   
   signal intAxisMaster_po : AxiStreamMasterArray(PIPES_G-1 downto 0);
   signal intAxisSlave_pi  : AxiStreamSlaveArray(PIPES_G-1 downto 0);

   --! 
   signal axisMaster_po : AxiStreamMasterArray(PIPES_G-1 downto 0);
   signal intPause : slv(PIPES_G-1 downto 0);

   --! Secondary clock and reset signals
   signal secClk: sl;
   signal secRst: sl;

   attribute keep : string;
   attribute keep of intAxisMaster_pi : signal is "true";
   attribute keep of intAxisSlave_po  : signal is "true";

---------------------------------------------------------------------------------------------------
begin
   
   --! Wire either one of the clock and reset inputs to output side of fifo
   GEN_ONE_CLK : if (TWO_CLOCKS_G = false) generate
      secClk   <= clk_i;
      secRst   <= rst_i;
   end generate GEN_ONE_CLK;

   GEN_TWO_CLKS : if (TWO_CLOCKS_G = true) generate
      secClk   <= sysClk_i;
      secRst   <= sysRst_i;
   end generate GEN_TWO_CLKS;
   

   u_RxFsm: entity work.RxFsm
   generic map (
      TPD_G      => TPD_G,
      PIPES_G    => PIPES_G,
      SOF_CHAR_G => SOF_CHAR_G,
      EOF_CHAR_G => EOF_CHAR_G,
      MAX_SIZE_G => MAX_BYTES_C)
   port map (
      clk_i        => clk_i,
      rst_i        => rst_i,
      axisMaster_o => intAxisMaster_po,
      axisSlave_i  => intAxisSlave_pi,
      pktFull_i    => pktFull,
      drop_o       => drop_o,
      dataRx_i     => dataRx_i,
      charRx_i     => charRx_i,
      link_i       => link_i);

   --! Axistream interconnect
   intAxisMaster_pi <= intAxisMaster_po;
   intAxisSlave_pi  <= intAxisSlave_po;

   --! PIPES_G number of fifos
   GEN_SYNC_FIFO :
   for i in PIPES_G-1 downto 0 generate
      --! CslAxisFifo module instantiation
      u_AxisFifo : entity work.CslAxisFifo
         generic map (
            TPD_G          => TPD_G,
            PIPELINE_G     => 2,
            BRAM_G         => false,
            AWIDTH_G       => FIFO_ADDR_WIDTH_C,
            THRESHOLD_G    => FIFO_PAUSE_THRESH_C,
            TUSER_NORMAL_G => false,
            TID_WIDTH_G    => 1,
            TDEST_WIDTH_G  => 1,
            TUSER_WIDTH_G  => 8,
            BYTES_SLV_G    => 1,
            BYTES_MST_G    => DATA_WIDTH_G
         )
         port map (
            sAxisClk_i    => clk_i,
            sAxisRst_i    => rst_i,
            sAxisMaster_i => intAxisMaster_pi(i),
            sAxisSlave_o  => intAxisSlave_po(i),
            mAxisClk_i    => secClk,
            mAxisRst_i    => secRst,
            mAxisMaster_o => axisMaster_po(i),
            mAxisSlave_i  => axisSlave_i(i),
            wrCnt_o       => fifoWrCnt(i),
            pause_o       => intPause(i)
         );

      --! Assign packet full from pause signal 
      pktFull(i)  <= intPause(i);
      --! Check packet empty. Empty if less words in fifo then BYTES in max packet
      pktEmpty(i) <= '1' when (unsigned(fifoWrCnt(i)(FIFO_ADDR_WIDTH_C-1 downto 0)) <= MAX_BYTES_C) 
                     else '0';
            
      --! Swap endianness because the axiStreamFifo is Little endian but we need big endian.
      axisMaster_o(i).tValid <= axisMaster_po(i).tValid;
      axisMaster_o(i).tStrb  <= axisMaster_po(i).tStrb;
      axisMaster_o(i).tKeep  <= axisMaster_po(i).tKeep;
      axisMaster_o(i).tLast  <= axisMaster_po(i).tLast;
      axisMaster_o(i).tDest  <= axisMaster_po(i).tDest;
      axisMaster_o(i).tId    <= axisMaster_po(i).tId;
      axisMaster_o(i).tUser(511 downto (DATA_WIDTH_G*8)) <= axisMaster_po(i).tUser(511 downto (DATA_WIDTH_G*8));      
      axisMaster_o(i).tUser((DATA_WIDTH_G*8)-1 downto 0) <= endianSwap(axisMaster_po(i).tUser((DATA_WIDTH_G*8)-1 downto 0));
      axisMaster_o(i).tData(511 downto (DATA_WIDTH_G*8)) <= axisMaster_po(i).tData(511 downto (DATA_WIDTH_G*8));
      axisMaster_o(i).tData((DATA_WIDTH_G*8)-1 downto 0) <= endianSwap(axisMaster_po(i).tData((DATA_WIDTH_G*8)-1 downto 0));
      --   
   end generate GEN_SYNC_FIFO;
   
   --! Assign the outputs
   pktEmpty_o <= pktEmpty;
   --
end rtl;
---------------------------------------------------------------------------------------------------