---------------------------------------------------------------------------------------------------
--! @brief     Fiber monitor module
--! @details    
--!   The module is an upgrade from StatusMon module. In this module GT signals are also monitored.
--!   The Watchdog module is used as Link OK module. Link is OK when no error are present from GT. 
--!    
--! @author Jernej Kokalj, Cosylab (jernej.kokalj@cosylab.com)
--!
--! @date Oct 15 2018 created
--! @date Jan 14 2019 last modify
--!
--! @version v0.1
--!
--! @file FiberMon.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use work.CslStdRtlPkg.all;
use work.FiberPkg.all;

library UNISIM;
use UNISIM.VComponents.all;
---------------------------------------------------------------------------------------------------
entity FiberMon is
   generic (
      TPD_G             : time      := 1 ns;
      STAT_SIZE_G       : positive  := 4;
      LINK_TIME_WIDTH_G : positive  := 32
   );
   port (
      clk_i             : in sl;                                     --! clock signal bus
      rst_i             : in sl;                                     --! reset signal bus
      clear_i           : in sl;                                     --! clear monitoring coutners signal
      statusBits_i      : in slv(STAT_SIZE_G-1 downto 0);            --! status bits vector input
      
      gtRx_i            : in GtType;                                  --! GT receiver signals
      linkTime_i        : in slv(LINK_TIME_WIDTH_G -1 downto 0);     --! link time for watchdog Link OK timer
      
      ilk_o             : out sl;                                    --! interlock output signal
      err_o             : out sl;                                    --! error output signal
      sts32array_o      : out Slv32Array(STAT_SIZE_G + 3 downto 0);  --! monitor counters output signal
      link_o            : out sl                                     --! link OK output signal
   );
end FiberMon;
---------------------------------------------------------------------------------------------------
architecture structure of FiberMon is

   -- internal signals
   signal statusBits    : slv(STAT_SIZE_G + 3 downto 0);
   signal wdRst         : sl;
---------------------------------------------------------------------------------------------------  
begin
   
   --! assigning status signal to statusBits vector
   statusBits(0)                          <= not gtRx_i.cdrStable; 
   statusBits(1)                          <= gtRx_i.decErr; 
   statusBits(2)                          <= gtRx_i.dispErr;
   statusBits(3)                          <= gtRx_i.byteReAlig;
   statusBits(STAT_SIZE_G + 3 downto 4)   <= statusBits_i;
      

   --! Status monitor module instantiation
   u0_StatusMon : entity work.StatusMon
      generic map (
         TPD_G  => TPD_G,
         SIZE_G => STAT_SIZE_G + 4
      )
      port map (
         clk_i        => clk_i,
         rst_i        => rst_i,
         clear_i      => clear_i,
         statusBits   => statusBits,
         ilk_o        => ilk_o,
         err_o        => err_o,
         sts32array_o => sts32array_o
      );
   
   --! Link OK reset signal => when any of GT errors is present
   wdRst <= gtRx_i.decErr or gtRx_i.dispErr or not(gtRx_i.rstDone) or 
            not(gtRx_i.byteAlig) or not(gtRx_i.cdrStable);   
     
   --! Watchdog module instantiation => used as Link OK 
   u0_Watchdog : entity work.Watchdog
      generic map (
         TPD_G   => TPD_G,
         WIDTH_G => LINK_TIME_WIDTH_G
      )
      port map (
         clk_i   => clk_i,
         rst_i   => rst_i,
         en_i    => '1',
         wdRst_i => wdRst,
         time_i  => linkTime_i,
         tout_o  => link_o
      );
   
end structure;
---------------------------------------------------------------------------------------------------