---------------------------------------------------------------------------------------------------
--! @brief  Fiber Link receiver control logic  
--! @details
--!   The control logic RxFsm parses the data stream from the fiber data byte stream, and sends the packets over the AxiStream pipes. 
--!   
--!   At the input of the logic the fiber data is sent through a 4-stage pipeline. 
--!   The pipeline is required so the CRC bytes and the EOF_G are already received and the CRC
--!   verified at the same time the last word of the message is received. 
--!   
--!   The 16-bit crc is calculated from polynomial: x^16 + x^12 + x^5 + 1
--!   
--! @author Uros Laget, Cosylab (uros.legat@cosylab.com)
--!
--! @date Sep 26 2018 created
--! @date Sep 26 2018 last modify
--!
--! @version v0.1
--!
--!
--! @file RxFsm.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;

use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;

--! @brief  
--! @details 
---------------------------------------------------------------------------------------------------
entity RxFsm is
generic (
      TPD_G          : time     := 1 ns;               --! Simulation delta time offset
      PIPES_G        : integer range 1 to 255 := 1;    --! Number of priority pipes (For HPD set to 1)        
      SOF_CHAR_G     : slv(7 downto 0) := x"1C";       --! Start of frame character
      EOF_CHAR_G     : slv(7 downto 0) := x"FE";       --! End of frame character
      MAX_SIZE_G     : positive := 8                   --! Maximum number of input axis data words 
   );
   port (
      clk_i         : in  sl;
      rst_i         : in  sl;

      --! User side interface

      --! Pipes
      axisMaster_o : out AxiStreamMasterArray(PIPES_G-1 downto 0);
      axisSlave_i  : in  AxiStreamSlaveArray(PIPES_G-1 downto 0);
      pktFull_i    : in slv(PIPES_G-1 downto 0);      
      drop_o       : out sl;    

      --! Fiber interface
      dataRx_i    : in slv(7 downto 0);
      charRx_i    : in sl;
      link_i      : in sl      
   );
end RxFsm;
---------------------------------------------------------------------------------------------------
architecture rtl of RxFsm is

   --! State type containing all states  
   type StateType is (
      IDLE_S,
      DROP_S,
      PIPE_S,
      MSG_S
   );
   
   --! Record containing all register elements
   type RegType is record
      --! Pipe number of axi stream masters
      axisMaster  : AxiStreamMasterArray(PIPES_G-1 downto 0);  
      --! Internal Counters
      msgCnt   : integer;
      pipe     : integer;
      --! Internal Errors
      drop     : sl;
      --! Enable CRC calculation from the data      
      crcEn    : sl;
      crcClr   : sl;
      --! Data Transmission byte
      dataRx   : Slv8Array(3 downto 0);
      charRx   : slv(3 downto 0);
      --
      state   : StateType;
   end record RegType;

   --! Initial and reset values for all register elements   
   constant REG_INIT_C : RegType := (
      --! Pipe number of slaves
      axisMaster  => (others => AXI_STREAM_MASTER_INIT_C),  
      --! Internal Counters
      msgCnt => 0,
      pipe   => 0,      
      --! Internal Errors
      drop   => '0',

      --! Enable CRC calculation from the data      
      crcEn    => '0',
      crcClr   => '1',
      --! Data Transmission byte
      dataRx   => (others => x"00"),
      charRx   => (others => '0'),
      ---
      state => IDLE_S
   );
   
   --! internal signals
   signal r      : RegType := REG_INIT_C;
   signal rin    : RegType;
   signal crc_po : slv(15 downto 0);
   
---------------------------------------------------------------------------------------------------
begin
   u_Crc16D8: entity work.Crc16D8
      generic map (
         TPD_G      => TPD_G)
      port map (
         clk_i   => clk_i,
         rst_i   => rst_i,
         data_i  => rin.dataRx(2),
         clear_i => rin.crcClr,
         en_i    => rin.crcEn,
         crc_o   => crc_po);

   --! @brief Combinational process
   --! @details Main module logic. Generates rin based on r and any module inputs
   comb : process (r, rst_i, axisSlave_i, crc_po, dataRx_i, charRx_i, link_i, pktFull_i) is
      variable v       : RegType;
      variable vCrcErr : sl;
   begin
      v := r;
      
      --! Data and char Rx pipeline
      v.dataRx(0) := dataRx_i;
      v.dataRx(1) := r.dataRx(0);
      v.dataRx(2) := r.dataRx(1); 
      v.dataRx(3) := r.dataRx(2);
      v.charRx(0) := charRx_i;
      v.charRx(1) := r.charRx(0);
      v.charRx(2) := r.charRx(1); 
      v.charRx(3) := r.charRx(2);
      
      --! Combinatorial crc check
      if (crc_po = r.dataRx(1) & r.dataRx(0)) then
         vCrcErr := '0';
      else
         vCrcErr := '1';   
      end if;
      
      --! State Machine
      case r.state is
         ----------------------------------------------------------------------
         when IDLE_S =>
            --! Put the message counter to 0
            --! Reset pipe
            v.msgCnt  := 0;
            v.pipe    := 0;
            
            --! CRC calculation not enabled, but cleared
            v.crcEn     := '0';
            v.crcClr    := '1';
            
            --! Put drop back to '0'
            v.drop      := '0';
            
            --! All streams in initial state
            v.axisMaster  := (others => AXI_STREAM_MASTER_INIT_C);

            --! Next state
            --! Look for the SOF_CHAR_G in the stream. Proceed only if link is ok.
            if (r.dataRx(2) = SOF_CHAR_G and r.charRx(2) = '1' and link_i='1') then
               v.crcEn     := '1';
               v.crcClr    := '0';           
               v.state := PIPE_S;
            end if;
         ----------------------------------------------------------------------
         when PIPE_S =>
            --! Put the message counter to 0
            --! Register the pipe from the data
            v.msgCnt  := 0;
            v.pipe    := to_integer(unsigned(r.dataRx(2)));
            
            --! CRC calculation enabled
            v.crcEn     := '1';
            v.crcClr    := '0';
            
            --! All streams in initial state
            v.axisMaster  := (others => AXI_STREAM_MASTER_INIT_C);

            --! Next state
            --! Check if pipe is valid, if fifo not full, and link is ok
            if (v.pipe    < PIPES_G and
                pktFull_i(v.pipe) = '0' and
                link_i    = '1'
            ) then
               v.state := MSG_S;
            else
               v.state := DROP_S;            
            end if;         
         ----------------------------------------------------------------------
         when DROP_S =>
            --! Put the message counter to 0
            --! Reset pipe
            v.msgCnt  := 0;
            v.pipe    := 0;
            
            --! CRC calculation not enabled, but cleared
            v.crcEn     := '0';
            v.crcClr    := '1';
            
             --! Drop the packet (stays only 1 c-c and should be caught by an external status counter)
            v.drop      := '1';
            
            --! All streams in initial state
            v.axisMaster  := (others => AXI_STREAM_MASTER_INIT_C);

            --! Next state go back to init
            v.state := IDLE_S;
      ----------------------------------------------------------------------
         when MSG_S =>
            --! Increment the message counter
            v.msgCnt  := r.msgCnt+1;
            
            --! CRC calculation enabled
            v.crcEn     := '1';
            v.crcClr    := '0';
            
            --! Move the data over axi stream
            --! Warning! This assumes that the fifo can still take the whole packet
            --! since the pktFull_i was low
            v.axisMaster(r.pipe).tValid            := '1';
            v.axisMaster(r.pipe).tLast             := '0';
            v.axisMaster(r.pipe).tData(7 downto 0) := r.dataRx(2);

            --! Next state
            --! Check if the EOF is in the pipe(0)
            --! The crc err is applied to tUser the length is good.
            if (v.dataRx(0)  = EOF_CHAR_G and
                v.charRx(0)  = '1'
            ) then
               v.axisMaster(r.pipe).tLast := '1';
               v.axisMaster(r.pipe).tUser(1 downto 0) := '0' & vCrcErr;               
               v.state := IDLE_S;
               
            --! If there is no EOF and the message reaches MAX_SIZE_G then the 
            --! the message is too long and the error is applied to tUser.
            --! The crc err is applied as well            
            elsif (r.msgCnt = MAX_SIZE_G-1) then
               v.axisMaster(r.pipe).tLast             := '1'; 
               v.axisMaster(r.pipe).tUser(1 downto 0) := '1' & vCrcErr;             
               v.state := IDLE_S;
            end if;             
         ----------------------------------------------------------------------
         when others =>
            --! This state should not be reached
            --! Initialize
            v := REG_INIT_C;
         ----------------------------------------------------------------------      
      end case;
      
      -- Reset
      if (rst_i = '1') then
         v := REG_INIT_C;
      end if;

      rin <= v;
      
      --! Output assignment
      axisMaster_o  <= r.axisMaster;
      drop_o        <= r.drop;    
   -----------------------------------------------
   end process comb;
   
   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk to create registers
   seq : process (clk_i) is
   begin
      if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
      end if;
   end process seq;
---------------------------------------------------------------------------------------------------
end rtl;
---------------------------------------------------------------------------------------------------