---------------------------------------------------------------------------------------------------
--! @brief  Fiber Link transmitter control logic  
--! @details
--!   The control logic TxFsm reads the scheduled packet from the FIFO, 
--!   adds the packet delimiter characters, pipe number, and CRC, and transmits the message to the fiber.
--!   The logic cycles through the pipes in round robin fashion and sends the packet if there is a message in the pipe.
--!   
--!   The logic checks for underrun error err_o(0) and length error err_o(1) and corrupts the crc error of the 
--!   fiber packet if detected.
--!   
--!   The 16-bit crc is calculated from polynomial: x^16 + x^12 + x^5 + 1
--!   
--! @author Uros Laget, Cosylab (uros.legat@cosylab.com)
--!
--! @date Sep 20 2018 created
--! @date Sep 20 2018 last modify
--!
--! @version v0.1
--!
--!
--! @file TxFsm.vhd
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;

use work.CslStdRtlPkg.all;
use work.CslAxiPkg.all;

--! @brief  
--! @details 
---------------------------------------------------------------------------------------------------
entity TxFsm is
generic (
      TPD_G          : time     := 1 ns;               --! Simulation delta time offset
      PIPES_G        : integer range 1 to 255 := 1;    --! Number of priority pipes (For HPD set to 1)        
      IDLE_G         : slv(8 downto 0) := '1' & x"BC"; --! Sent out at when IDLE_G
      SOF_CHAR_G     : slv(7 downto 0) := x"1C";       --! Start of frame character
      EOF_CHAR_G     : slv(7 downto 0) := x"FE";       --! End of frame character
      MAX_SIZE_G     : positive := 8                   --! Maximum number of input axis data words 
   );
   port (
      clk_i         : in  sl;
      rst_i         : in  sl;

      --! User side interface

      --! Pipes
      axisMaster_i : in  AxiStreamMasterArray(PIPES_G-1 downto 0);
      axisSlave_o  : out AxiStreamSlaveArray(PIPES_G-1 downto 0);      
      err_o        : out Slv2Array(PIPES_G-1 downto 0);    

      --! Fiber interface
      dataTx_o    : out slv(7 downto 0);
      charTx_o    : out sl;
      idle_o      : out sl
   );
end TxFsm;
---------------------------------------------------------------------------------------------------
architecture rtl of TxFsm is

   --! State type containing all states  
   type StateType is (
      IDLE_S,
      SOF_S,
      PIPE_S,
      MSG_S,
      CRC0_S,
      CRC1_S,
      EOF_S,
      FLUSH_S
   );
   
   --! Record containing all register elements
   type RegType is record
      --! Pipe number of slaves
      axisSlave  : AxiStreamSlaveArray(PIPES_G-1 downto 0);  
      --! Internal Counters
      pipeCnt  : integer;
      msgCnt   : integer;
      --! Internal Errors
      lenErr   : slv(PIPES_G-1 downto 0);
      urnErr   : slv(PIPES_G-1 downto 0);
      --! Enable CRC calculation from the data      
      crcEn    : sl;
      crcClr    : sl;
      --! Data Transmission byte
      dataTx   : slv(7 downto 0);
      charTx   : sl;
      idle     : sl;
      --
      state   : StateType;
   end record RegType;

   --! Initial and reset values for all register elements   
   constant REG_INIT_C : RegType := (
      --! Pipe number of slaves
      axisSlave  => (others => AXI_STREAM_SLAVE_INIT_C),  
      --! Internal Counters
      pipeCnt  => 0,
      msgCnt   => 0,
      --! Internal Errors
      lenErr   => (others => '0'),
      urnErr   => (others => '0'),
      --! Enable CRC calculation from the data      
      crcEn    => '0',
      crcClr   => '1',
      --! Data Transmission byte
      dataTx   => (others => '0'),
      charTx   => '0',
      idle     => '0',
      ---
      state => IDLE_S
   );
   
   --! internal signals
   signal r      : RegType := REG_INIT_C;
   signal rin    : RegType;
   signal crc_po : slv(15 downto 0);
   
---------------------------------------------------------------------------------------------------
begin
   u_Crc16D8: entity work.Crc16D8
      generic map (
         TPD_G      => TPD_G)
      port map (
         clk_i   => clk_i,
         rst_i   => rst_i,
         data_i  => rin.dataTx,
         clear_i => rin.crcClr,
         en_i    => rin.crcEn,
         crc_o   => crc_po);

   --! @brief Combinational process
   --! @details Main module logic. Generates rin based on r and any module inputs
   comb : process (r, rst_i, axisMaster_i, crc_po) is
      variable v    : RegType;
   begin
      v        := r;
      v.idle   := '0';
       -- State Machine
      case r.state is
         ----------------------------------------------------------------------
         when IDLE_S =>
            v.idle   := '1';
            --! All streams not ready
            v.axisSlave  := (others => AXI_STREAM_SLAVE_INIT_C);
            
            --! Increment revolving pipe counter
            --! This makes the round robin selection of the pipes
            if (r.pipeCnt < PIPES_G-1) then
               v.pipeCnt   := r.pipeCnt + 1;
            else
               v.pipeCnt   := 0;
            end if;
            v.msgCnt    := 0;
            
            --! CRC calculation not enabled, but cleared
            --! Errors hold the last value
            v.crcEn     := '0';
            v.crcClr    := '1';
            
            --! Data Transmission byte
            v.dataTx    := IDLE_G(7 downto 0);
            v.charTx    := IDLE_G(8);

            --! Next state
            --! Look at the next value (v.) so the r.pipeCnt 
            --! can be statically addressed in next states 
            if (axisMaster_i(v.pipeCnt).tValid = '1') then
                  v.state := SOF_S;
            end if;
         ----------------------------------------------------------------------
         when SOF_S =>
            --! All streams not ready
            v.axisSlave  := (others => AXI_STREAM_SLAVE_INIT_C);
            
            --! Message counter still 0
            --! r.pipeCnt holds the last value and is fixed throughout the rest of the FSM
            v.msgCnt    := 0;
            
            --! CRC calculation not enabled
            --! Errors of the addressed pipe are cleared
            v.crcEn  := '0';
            v.crcClr := '0';
            v.lenErr(r.pipeCnt) := '0';            
            v.urnErr(r.pipeCnt) := '0';    
            
            --! Data Transmission byte
            v.dataTx    := SOF_CHAR_G;
            v.charTx    := '1';

            --! Next state
            v.state := PIPE_S;         
         ----------------------------------------------------------------------
         when PIPE_S =>
            --! All streams not ready
            v.axisSlave := (others => AXI_STREAM_SLAVE_INIT_C);
            
            --! Message counter still 0
            --! r.pipeCnt holds the last value and is fixed throughout the rest of the FSM
            v.msgCnt    := 0;
            
            --! CRC calculation not enabled
            --! Errors of the addressed pipe are cleared
            v.crcEn             := '1';
            v.lenErr(r.pipeCnt) := '0';            
            v.urnErr(r.pipeCnt) := '0';    
            
            --! Data Transmission byte
            v.dataTx    := toSlv(r.pipeCnt,8);
            v.charTx    := '0';

            --! Next state
            v.state := MSG_S;
      ----------------------------------------------------------------------
         when MSG_S =>
            --! Enable movement of the stream of the addressed pipe 
            v.axisSlave(r.pipeCnt) := AXI_STREAM_SLAVE_FORCE_C;
            
            --! Message counter still 0
            --! r.pipeCnt holds the last value and is fixed throughout the rest of the FSM
            v.msgCnt    := r.msgCnt + 1;
            
            --! CRC calculation enabled
            --! Errors of the addressed pipe are checked
            v.crcEn     := '1';
            
            --! The length must not exceed MAX_SIZE_G        
            if ((r.msgCnt = MAX_SIZE_G-1 and  axisMaster_i(r.pipeCnt).tLast/='1') or
                r.msgCnt > MAX_SIZE_G-1            
            ) then            
               v.lenErr(r.pipeCnt) := '1';
            end if;

            --! The data valid must not drop otherwise under-run err
            if (axisMaster_i(r.pipeCnt).tValid = '0') then            
               v.urnErr(r.pipeCnt) := '1';
            end if;               
            
            
            v.urnErr(r.pipeCnt) := '0';    
            
            --! Data Transmission byte
            v.dataTx    := axisMaster_i(r.pipeCnt).tData(7 downto 0);
            v.charTx    := '0';

            --! Next state condition
            if (r.msgCnt = MAX_SIZE_G-1 or axisMaster_i(r.pipeCnt).tLast='1') then
               v.state := CRC0_S;
            end if;
         ----------------------------------------------------------------------
         when CRC0_S =>
            --! All streams not ready
            v.axisSlave := (others => AXI_STREAM_SLAVE_INIT_C);
            
            --! Hold the message counter value
            --! r.pipeCnt holds the last value and is fixed throughout the rest of the FSM
            
            --! Disable CRC calculation
            --! Errors of the addressed pipe hold the value
            v.crcEn             := '0';
           
            --! Data Transmission byte
            --! If any error occurs the crc_po is inverted corrupted. 
            --! This ensures that the receiving side will detect the transmission error
            v.dataTx    := crc_po(15 downto 8);
            if (r.urnErr(r.pipeCnt) = '1' or r.lenErr(r.pipeCnt) = '1') then
               v.dataTx := not crc_po(15 downto 8);
            end if;
            v.charTx := '0';

            --! Next state
            v.state := CRC1_S;
         ----------------------------------------------------------------------
         when CRC1_S =>
            --! All streams not ready
            v.axisSlave := (others => AXI_STREAM_SLAVE_INIT_C);
            
            --! Hold the message counter value
            --! r.pipeCnt holds the last value and is fixed throughout the rest of the FSM
            
            --! Disable CRC calculation
            --! Errors of the addressed pipe hold the value
            v.crcEn             := '0';
           
            --! Data Transmission byte
            --! If any error occurs the crc_po is inverted corrupted. 
            --! This ensures that the receiving side will detect the transmission error
            v.dataTx    := crc_po(7 downto 0);
            if (r.urnErr(r.pipeCnt) = '1' or r.lenErr(r.pipeCnt) = '1') then
               v.dataTx := not crc_po(7 downto 0);
            end if;
            v.charTx := '0';

            --! Next state
            v.state := EOF_S;            
         ----------------------------------------------------------------------
         when EOF_S =>
            --! All streams not ready
            v.axisSlave  := (others => AXI_STREAM_SLAVE_INIT_C);
            
            --! Hold the message counter value
            --! r.pipeCnt holds the last value and is fixed throughout the rest of the FSM
            
            --! Disable CRC calculation
            --! Errors of the addressed pipe hold the value
            v.crcEn     := '0';   
            
            --! Data Transmission byte
            v.dataTx    := EOF_CHAR_G;
            v.charTx    := '1';

            --! Next state
            if (r.lenErr(r.pipeCnt) = '1' ) then
               v.state := FLUSH_S;
            else 
               v.state := IDLE_S;
            end if;
         ----------------------------------------------------------------------
         when FLUSH_S =>
            --! If length error occurred the rest of the packet is still in the FIFO 
            --! So the fifo has to be flushed out until the end of the packet tLast
            v.axisSlave(r.pipeCnt) :=  AXI_STREAM_SLAVE_FORCE_C;
            
            --! Hold the message counter value
            --! r.pipeCnt holds the last value and is fixed throughout the rest of the FSM
            
            --! Disable CRC calculation
            --! Errors of the addressed pipe hold the value
            v.crcEn     := '0';   
            
            --! Data Transmission byte
            v.dataTx    := IDLE_G(7 downto 0);
            v.charTx    := IDLE_G(8);

            --! Next state
            if (axisMaster_i(r.pipeCnt).tLast ='1' and 
                axisMaster_i(r.pipeCnt).tValid='1') then
               v.state := IDLE_S;
            end if;
         ----------------------------------------------------------------------
         when others =>
            --! This state should not be reached
            --! Initialize
            v := REG_INIT_C;
         ----------------------------------------------------------------------      
      end case;
      
      -- Reset
      if (rst_i = '1') then
         v := REG_INIT_C;
      end if;

      rin <= v;
      
      --! Output assignment
      axisSlave_o  <= v.axisSlave;

      --! Loop assign the Errors
      for I in PIPES_G-1 downto 0 loop
         err_o(i)(0)     <= r.urnErr(i);
         err_o(i)(1)     <= r.lenErr(i);
      end loop;      
            
      --! Fiber i      
      dataTx_o     <= r.dataTx;   
      charTx_o     <= r.charTx;    
      idle_o       <= r.idle;
   -----------------------------------------------
   end process comb;
   
   --! @brief Sequential process
   --! @details Assign rin to r on rising edge of clk to create registers
   seq : process (clk_i) is
   begin
      if (rising_edge(clk_i)) then
         r <= rin after TPD_G;
      end if;
   end process seq;
---------------------------------------------------------------------------------------------------
end rtl;
---------------------------------------------------------------------------------------------------