---------------------------------------------------------------------------------------------------
--! @brief Fiber Link transmitter test bench 
--! @details
--!
--!
--! @author Uros Laget, Cosylab (uros.legat@cosylab.com)
--!
--! @date Sep 20 2018 created
--! @date Sep 20 2018 last modify
--!
--! @version v0.1
--!
--!
--! @file FiberLinkTx.vhd
---------------------------------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;

use work.CslStdRtlPkg.all;
use work.AxiStreamPkg.all;

entity  FiberLinkTxTb is

end entity ;
--------------------------------------------------------------------------------


architecture Bhv of FiberLinkTxTb is
  -----------------------------
  -- Port Signals 
  -----------------------------
   -- Clock reset
   constant CLK_PERIOD_C      : time    := 10 ns;
   constant CLK_DELAY_C       : time    := 0 ns;
   constant RST_START_DELAY_C : time    := 1 ns; 
   constant RST_HOLD_TIME_C   : time    := 100 ns;
   constant SYNC_RESET_C      : boolean := true;
   --
   constant TPD_C        : time                    := 1 ns;
   constant DATA_WIDTH_C : natural                 := 4;
   constant PIPES_C      : integer range 1 to 255  := 4;
   constant IDLE_C       : slv(8 downto 0)         := '1' & x"BC";
   constant SOF_CHAR_C   : slv(7 downto 0)         := x"1C";  
   constant EOF_CHAR_C   : slv(7 downto 0)         := x"FE";  
   constant MAX_SIZE_C   : positive                := 8;
   --
   signal   clk_i        : sl:= '0';
   signal   rst_i        : sl:= '1';
   --
   signal   strobe_i     : slv(PIPES_C-1 downto 0):= x"0";  
   signal   msg32Array_i : Slv32Array(MAX_SIZE_C-1 downto 0):= (others=> x"01234567");
   
   signal   inputMaster  : AxiStreamMasterArray(PIPES_C-1 downto 0):=(others => AXI_STREAM_MASTER_INIT_C);
   signal   inputSlave   : AxiStreamSlaveArray(PIPES_C-1 downto 0);
   signal   outputMaster : AxiStreamMasterArray(PIPES_C-1 downto 0):=(others => AXI_STREAM_MASTER_INIT_C);
   signal   outputSlave  : AxiStreamSlaveArray(PIPES_C-1 downto 0);
   signal   txPktFull_o  : slv(PIPES_C-1 downto 0);
   signal   txErr_o      : Slv2Array(PIPES_C-1 downto 0);
   signal   dataTx_o     : slv(7 downto 0);
   signal   charTx_o     : sl;
   signal   pktEmpty_o  : slv(PIPES_C-1 downto 0);
   
   --! 
   signal   msgDrop_o     : slv(PIPES_C-1 downto 0);
   signal   drop_o        : sl;
begin  -- architecture Bhv

   -----------------------------
   -- component instantiation 
   -----------------------------

   --! Message transmitters
   GEN_PIPES :
   for i in PIPES_C-1 downto 0 generate
      u_MsgStreamTx: entity work.MsgStreamTx
      generic map (
         TPD_G  => TPD_C,
         SIZE_G => MAX_SIZE_C)
      port map (
         clk_i        => clk_i,
         rst_i        => rst_i,
         strobe_i     => strobe_i(i),
         msg32Array_i => msg32Array_i,
         full_i       => txPktFull_o(i),
         axisMaster_o => inputMaster(i),
         axisSlave_i  => inputSlave(i),
         drop_o       => msgDrop_o(i));
   end generate GEN_PIPES;

   u_FiberLinkTx: entity work.FiberLinkTx
   generic map (
      TPD_G        => TPD_C,
      DATA_WIDTH_G => DATA_WIDTH_C,
      PIPES_G      => PIPES_C,
      IDLE_G       => IDLE_C,
      SOF_CHAR_G   => SOF_CHAR_C,
      EOF_CHAR_G   => EOF_CHAR_C,
      MAX_SIZE_G   => MAX_SIZE_C)
   port map (
      clk_i        => clk_i,
      rst_i        => rst_i,
      axisMaster_i => inputMaster,
      axisSlave_o  => inputSlave,
      txPktFull_o  => txPktFull_o,
      txErr_o      => txErr_o,
      dataTx_o     => dataTx_o,
      charTx_o     => charTx_o);
   
   
   u_FiberLinkRx: entity work.FiberLinkRx
   generic map (
      TPD_G        => TPD_C,
      DATA_WIDTH_G => DATA_WIDTH_C,
      PIPES_G      => PIPES_C,
      IDLE_G       => IDLE_C,
      SOF_CHAR_G   => SOF_CHAR_C,
      EOF_CHAR_G   => EOF_CHAR_C,
      MAX_SIZE_G   => MAX_SIZE_C)
   port map (
      clk_i        => clk_i,
      rst_i        => rst_i,
      dataRx_i     => dataTx_o,
      charRx_i     => charTx_o,
      link_i       => '1',
      axisMaster_o => outputMaster,
      axisSlave_i  => outputSlave,
      pktEmpty_o   => pktEmpty_o,
      drop_o       => drop_o);
   

   -- Pipes 
   GEN_PIPES0 :
   for i in PIPES_C-1 downto 0 generate
      --outputSlave(i) <= AXI_STREAM_SLAVE_INIT_C;
      outputSlave(i).tReady <= '1' when pktEmpty_o(i)='0' and rst_i='0' else '0';
   end generate GEN_PIPES0;
   
   
   -- Clock
	process is
   begin
      wait for CLK_DELAY_C;
      while (true) loop
         clk_i <= not clk_i;
         wait for CLK_PERIOD_C/2.0;
      end loop;
   end process;
   
   -- Reset
   process is
   begin
      rst_i <= '1';
      wait for RST_START_DELAY_C;
      if (SYNC_RESET_C) then
         wait until clk_i = '1';
      end if;
         rst_i <= '1';
         wait for RST_HOLD_TIME_C;
      if (SYNC_RESET_C) then
         wait until clk_i = '1';
      end if;
         rst_i <= '0';
      wait;
   end process;
   
   
   StimuliProcess : process
   begin
     -- wait until rst_i = '0';
      msg32Array_i <= (
         0 => x"01234567",
         1 => x"89ABCDEF",      
         2 => x"01234567",
         3 => x"89ABCDEF",
         4 => x"01234567",
         5 => x"89ABCDEF",      
         6 => x"01234567",
         7 => x"89ABCDEF"
       --  8 => x"DEADBEEF"
      );
         
      wait for 10 * RST_HOLD_TIME_C;
      wait for 5*CLK_PERIOD_C;
      wait until clk_i = '1';
      strobe_i <= x"9";
      wait until clk_i = '1';
      strobe_i <= x"0";
      wait for RST_HOLD_TIME_C;
      wait for 3*CLK_PERIOD_C;
      wait until clk_i = '1';
      strobe_i <= x"6";
      wait until clk_i = '1';
      strobe_i <= x"0";
      --wait;
     -- wait for 10*CLK_PERIOD_C;

      
   end process StimuliProcess;
  
end architecture Bhv;