from PyBta.Bta import LibParam
from PyBta.Bta import QueueMode
from PyBta.PyFrame import PyBtaFrameMode
from PyBta.PyFrame import PyBtaDataFormat
from PyBta.PyFrame import PyBtaChannelId
from PyBta.BtaWrapper import BtaWrapper
from PyBta.BtaWrapper import BtaException
import numpy as np
import scipy.constants as constant
import traceback as tb
import cv2
import time
import sys


def main():
    py_bta = BtaWrapper
    msg = py_bta.init()
    print(msg)

    try:

        # open a bltstream:
        py_bta.openBltstream('test.bltstream', 1, QueueMode.DropOldest, None)
        # Read a libParam like this:
        frameCount = py_bta.get_lib_param(LibParam.BltStreamTotalFrameCount)
        print(f"Opened bltstream with {frameCount} frames")
        frmInd = 0.0
        for frmInd in np.arange(0.0, frameCount, 1.0):
            py_bta.set_lib_param(LibParam.BltStreamPos, frmInd)
            frame = py_bta.get_frame(500)
            # Do something with the frame.. let's print the frame counter at least
            print(f"frame index {frmInd}: {frame.frameCounter}")
            show_image(frame)
            time.sleep(1) # Only showing a few frames, so let's saviour each one of them
        
        # close in order to clean up BltTofApi
        py_bta.close()


        # connect to an eth device
        #py_bta.openEth('192.168.0.10', None, 10001, False, '224.0.0.1', 10002, 5, QueueMode.DropOldest, None);
        py_bta.openEth('192.168.0.10', None, 10001, True, None, None, 5, QueueMode.DropOldest, None);
        # Method for reading integration time register
        intTime = py_bta.get_integration_time()
        print("Integration time: {0}", intTime)
        # Method for reading register(s) by address
        registers = py_bta.read_register(1, 10)
        print("Registers 1 through 10:")
        for register in registers:
            print(register)

        frameModes = [ PyBtaFrameMode.DistAmp, PyBtaFrameMode.ZAmp, PyBtaFrameMode.DistAmpFlags, PyBtaFrameMode.XYZ, PyBtaFrameMode.XYZAmp, PyBtaFrameMode.DistAmpColor, PyBtaFrameMode.XYZAmpFlags, PyBtaFrameMode.Intensities, \
                       PyBtaFrameMode.DistColor, PyBtaFrameMode.DistAmpBalance, PyBtaFrameMode.XYZColor, PyBtaFrameMode.Dist, PyBtaFrameMode.Amp, PyBtaFrameMode.RawdistAmp, PyBtaFrameMode.XYZConfColor, PyBtaFrameMode.DistAmpConf
                     ]
        for frameMode in frameModes:
            print()
            print(f"Setting frame mode {frameMode.name}")
            try:
                py_bta.set_frame_mode(frameMode)
                time.sleep(3) # Make sure the changes are fully applied and we don't get old frames from last frame mode
            except BtaException as e:
                print(f"{e}")
                continue
            print("Getting some frames:")
            for x in range(1, 201):
                # Method for accepting a frame from the BltTofApi
                frame = py_bta.get_frame(3000)
                # Do something with the frame.. let's print the frame counter at least
                print(frame.frameCounter, end = " ")
                sys.stdout.flush()
                show_image(frame)
            print()

        # close in order to clean up BltTofApi
        py_bta.close()

    except BtaException as e:
        print(f"exception: {e}")
        tb.print_exc()


def show_image(frame):
    for channel in frame.channels:
        if channel.id == PyBtaChannelId.Distance and channel.dataFormat == PyBtaDataFormat.UInt16:
            # This is an uint16 distance channel. Pixels with a value < 10 (may differ for older models, please consult your manual) are treated as invalid. It is displayd with the jet color scale
            # Reshape channel
            channelToShow = channel.data.reshape(channel.yRes, channel.xRes, 1)
            channelToShow[channelToShow < 10] = 0
            # Scale values into the range from [0mm - unabiguous range] to [0 - 255]
            mod_freq = channel.modulationFrequency
            unambigousrange_mm = (constant.speed_of_light/(2*mod_freq))*1000
            channelToShow[channelToShow > unambigousrange_mm] = unambigousrange_mm
            channelToShow = (channelToShow*255.0/unambigousrange_mm).astype('uint8')
            channelToShow = cv2.applyColorMap(channelToShow, cv2.COLORMAP_JET)
            # Invalid pixels with value 0 are mapped to color (128, 0, 0). But we want them to be black!
            channelToShow[np.all(channelToShow == (128, 0, 0), axis = -1)] = (0, 0, 0)
            cv2.imshow('Distance', channelToShow)
            cv2.waitKey(1)

        elif channel.id == PyBtaChannelId.Amplitude and channel.dataFormat == PyBtaDataFormat.UInt16:
            # This is an uint16 amplitude channel. All pixels of all amplitudes are valid values. It is displayd as a monochrome image
            # Reshape channel
            channelToShow = channel.data.reshape(channel.yRes, channel.xRes, 1)
            # Scale values into the range from [0 - amp_scale] to [0 - 255]
            amp_scale = 2 * np.average(channelToShow)
            channelToShow[channelToShow > amp_scale] = amp_scale
            channelToShow = (channelToShow*255.0/amp_scale).astype('uint8')
            cv2.imshow('Amplitude', channelToShow)
            cv2.waitKey(1)

        elif (channel.id == PyBtaChannelId.X or \
              channel.id == PyBtaChannelId.Y or \
              channel.id == PyBtaChannelId.Z) and \
             channel.dataFormat == PyBtaDataFormat.SInt16:
            sideLen = 2500
            # This is a int16 X, Y or Z coordinates channel. Pixels with a value < -32758 (may differ for older models, please consult your manual) are treated as invalid. It is displayd with the jet color scale
            # Reshape channel
            channelToShow = channel.data.reshape(channel.yRes, channel.xRes, 1)
            # Scale values into the range from [-<sideLen>mm - <sideLen>mm] to [0 - 255] (we only look at a cubic scene of 2x sideLen. Anything outside or invalid will be black
            channelToShow[channelToShow < -sideLen] = -sideLen
            channelToShow[channelToShow > sideLen] = sideLen
            channelToShow = ((channelToShow + sideLen)*255.0/sideLen/2.0).astype('uint8')
            channelToShow = cv2.applyColorMap(channelToShow, cv2.COLORMAP_JET)
            # Invalid pixels outside our 2x <sideLen> are mapped to color (128, 0, 0) or (0, 0, 128). But we want them to be black!
            channelToShow[np.all(channelToShow == (128, 0, 0), axis = -1)] = (0, 0, 0)
            channelToShow[np.all(channelToShow == (0, 0, 128), axis = -1)] = (0, 0, 0)
            cv2.imshow(str(channel.id)[-1], channelToShow)
            cv2.waitKey(1)

        elif channel.id == PyBtaChannelId.RawDist and channel.dataFormat == PyBtaDataFormat.UInt16:
            # This is an uint16 raw distance channel. All pixels are valid. The unambiguous range is mapped to the value range of uint16. It is displayd with the jet color scale
            # Reshape channel
            channelToShow = channel.data.reshape(channel.yRes, channel.xRes, 1)
            channelToShow = (channelToShow*255.0/65535.0).astype('uint8')
            channelToShow = cv2.applyColorMap(channelToShow, cv2.COLORMAP_JET)
            cv2.imshow('Raw Distance', channelToShow)
            cv2.waitKey(1)

        elif channel.id == PyBtaChannelId.Confidence and channel.dataFormat == PyBtaDataFormat.UInt8:
            # This is an uint8 confidence channel. It is displayd as a monochrome image
            # Reshape channel
            channelToShow = channel.data.reshape(channel.yRes, channel.xRes, 1)
            # No scaling, channel is already 8 bit per pixel
            cv2.imshow('Confidence', channelToShow)
            cv2.waitKey(1)

        elif channel.id == PyBtaChannelId.Color and channel.dataFormat == PyBtaDataFormat.Yuv422:
            # This is a YUV color channel. It is displayd as is
            # Reshape channel
            channelToShow = channel.data.reshape(channel.yRes, channel.xRes, 2)
            channelToShow = cv2.cvtColor(channelToShow, cv2.COLOR_YUV2BGR_UYVY)
            cv2.imshow('Color lens' + str(channel.lensIndex), channelToShow)
            cv2.waitKey(1)

        elif channel.id == PyBtaChannelId.Color and channel.dataFormat == PyBtaDataFormat.Yuv444:
            # This is a YUV color channel. It is displayd as is
            # Reshape channel
            channelToShow = channel.data.reshape(channel.yRes, channel.xRes, 3)
            channelToShow = cv2.cvtColor(channelToShow, cv2.COLOR_YUV2BGR)
            cv2.imshow('Color lens' + str(channel.lensIndex), channelToShow)
            cv2.waitKey(1)

        elif channel.id == PyBtaChannelId.Color and channel.dataFormat == PyBtaDataFormat.Yuv444UYV:
            # This is a YUV color channel. It is displayd as is
            # Reshape channel
            channelToShow = channel.data.reshape(channel.yRes, channel.xRes, 3)
            (U, Y, V) = cv2.split(channelToShow)
            channelToShow = cv2.merge([Y, U, V])
            channelToShow = cv2.cvtColor(channelToShow, cv2.COLOR_YUV2BGR)
            cv2.imshow('Color lens' + str(channel.lensIndex), channelToShow)
            cv2.waitKey(1)

        else:
            print(f"ChannelId {channel.id} with DataFormat {channel.dataFormat} not supported")



if __name__ == "__main__":
    main()
