/** *********************************************************************
 * @file flash.h
 * @author Daniel Weber, Roland Oberhammer
 * @date 2005-10-25
 * @version 1.0
 *
 * provides flash utility functions
 *
 *
 * @see flash.c
  ********************************************************************* **/
  
/** @ingroup lowleveldrv_grp
         
   
   @{
*/

/**
	@name Flash Utility Functions for Intel Strata Compatible Devices
*/
//@{
#ifndef _FLASH_H_INCLUDED_
#define _FLASH_H_INCLUDED_

#include "GPIOconfig.h"

#define FLASH_PROGRAM_TIMEOUT 			0x0ffffff
#define FLASH_ERASE_TIMEOUT				0x0ffffff
#define FLASH_UNLOCK_TIMEOUT			0x0ffffff

#define FLASH_DEBUG_LEVEL				0
#define FLASH_MAX_NOF_BANK_SELECT		4

#define FLASH_LAST_SECTOR				(FLASH_SECTOR_SIZE * (FLASH_NUM_SECTORS - 1))
#define FLASH_SIZE						(FLASH_SECTOR_SIZE * FLASH_NUM_SECTORS)


#ifdef _USE_VDK_

#include <VDK.h>

//prototype declarations for multithreading
char Flash_LockAccess (VDK_ThreadID *ThreadId);
char Flash_UnlockAccess (void);

#endif

typedef struct {
	unsigned char cNofBanks;
	unsigned long nBankSize;
	unsigned char cNofBankSelect;
	T_GPIO_MASK nBankSelect[FLASH_MAX_NOF_BANK_SELECT];
#ifdef _USE_VDK_
	bool bLocked;
	VDK_ThreadID tThreadId;
#endif
	unsigned char cCurrentBank;
} T_FLASH_SPEC;

//prototype declarations.
/**
	Sets some global configuration variables for managment purposes.
	Has to called prior to use the flash.
*/
void Flash_Setup(void);

/**
	@param startAddress 	start address of the flash device
	@param sectorAddress	address of the sector to be erased
	@return 				content of the flash status register (on success 0x0080 for Intel Strata Compatible Devices)
	
	erases the specified sector
	
*/
unsigned short Flash_EraseSector(unsigned long startAddress, unsigned long sectorAddress);

/**
	@return content of the flash status register (on success 0x0080 for Intel Strata Compatible Devices)
	
	erases the entire flash
*/
unsigned short Flash_EraseChip(void);

/**
	@param startAddress 	start address of the flash device
	@param wordAddress		address of the word to be programmed
	@param data				value to be written
	@return 				content of the flash status register (on success 0x0080 for Intel Strata Compatible Devices)
	
	programms a word at the specified address
*/
unsigned short Flash_ProgramWord(unsigned long startAddress, unsigned long wordAddress, unsigned short data);

/**
	@param cDevCode			holds the device code after the function call
	@param cManuCode		holds the manufacturer code after the function call
	
	reads flash device and manufacturer codes
*/
void Flash_ReadIdentifier (unsigned short *cDevCode, unsigned short *cManuCode);

/**
	@param pa_nSectorAddr	sector address to be locked
	@return					content of the flash status register (on success 0x0080 for Intel Strata Compatible Devices)
	
	locks a specified sector of the flash
*/
unsigned short Flash_LockBlock (unsigned long pa_nSectorAddr);

/**
	@param pa_nSectorAddr	sector address to be unlocked
	@return					content of the flash status register (on success 0x0080 for Intel Strata Compatible Devices)
	
	locks a specified sector of the flash
*/
unsigned short Flash_UnlockBlock (unsigned long pa_nSectorAddr);

/**
	@return content of the flash status register (on success 0x0080 for Intel Strata Compatible Devices)
	
	resets the flash's state machine to default values
*/
unsigned short Flash_Reset (void);

/**
	@param pa_nOffset 	address offset to be read
	@return 			content of the flash status register (on success 0x0080 for Intel Strata Compatible Devices)
	
	reads a word from the specified address
*/
unsigned short Flash_ReadWord (unsigned long pa_nOffset);

/**
	@param ulOffset		address offset
	@return 			sector number containing the address offset
	
	calculates a sector number for a given address offset
*/
int Flash_GetSectorNumber( unsigned long ulOffset);

/**
	@param pa_nStartAddress	start address of the check
	@return 				true if empty, otherwise false
*/
bool Flash_CheckIfEmpty (unsigned long pa_nStartAddress);

#endif

//@}
//@}


