///////////////////////////////////////////////////////////////
//
// BF533EzFlash.c
//
// Analog Devices, Inc. - 2001
//
//
// Change Log
//
//		1.00.1
//			- made changes so that the driver will work with
//			  the revised GUI
//
//		1.00.0
//			- initial release
//		modified by suedtirolli (C) 2005
//
// VisualDSP++ "Flash Programmer" flash driver for use with the
// CM-BF537 provided by Bluetechnix.
//
///////////////////////////////////////////////////////////////

// error enum
#include <Environment.h>
#include "Errors.h"
#include "../../../../driver/src/common/flash.h"

// #defines
#define BUFFER_SIZE		0x3000


// structure for flash sector information
typedef struct _SECTORLOCATION{
	long lStartOff;
	long lEndOff;
}SECTORLOCATION;


// Flash Programmer commands
typedef enum
{
	NO_COMMAND,		// 0
	GET_CODES,		// 1
	RESET_FLASH,	// 2
	WRITE,			// 3
	FILL,			// 4
	ERASE_ALL,		// 5
	ERASE_SECT,		// 6
	READ,			// 7
	GET_SECTNUM,	// 8
	GET_SECSTARTEND	// 9
}enProgCmds;

// #defines for the assembler
asm ("#define FLASH_START_L 0x0000");
asm ("#define FLASH_START_H 0x2000");

// function prototypes
ERROR_CODE SetupForFlash();
ERROR_CODE GetCodes();
ERROR_CODE PollToggleBit(unsigned long ulOffset);
ERROR_CODE ResetFlash();
ERROR_CODE EraseFlash();
ERROR_CODE EraseBlock( int nBlock );
ERROR_CODE UnlockFlash(unsigned long ulOffset);
ERROR_CODE WriteData( unsigned long ulStart, long lCount, long lStride, int *pnData );
ERROR_CODE FillData( unsigned long ulStart, long lCount, long lStride, int *pnData );
ERROR_CODE ReadData( unsigned long ulStart, long lCount, long lStride, int *pnData );
ERROR_CODE ReadFlash( unsigned long ulOffset, unsigned short *pnValue );
ERROR_CODE WriteFlash( unsigned long ulOffset, int nValue );
ERROR_CODE GetSectorNumber( unsigned long ulOffset, int *pnSector );
ERROR_CODE GetSectorStartEnd( long *lStartOff, long *lEndOff, int nSector );

// global data for use with the VisualDSP++ plug-in
char 			*AFP_Title = "CM-BF548";
char 			*AFP_Description = "Intel P30";
enProgCmds 		AFP_Command = NO_COMMAND;
char			*AFP_DeviceCompany = "Intel";		// Flash Company
char 			*AFP_DrvVersion		= "1.01.0";		// Driver Version
char			*AFP_BuildDate		= __DATE__;		// Driver Build Date
int 			AFP_ManCode = -1;				
int 			AFP_DevCode = -1;				
unsigned long 	AFP_Offset = 0x0;
int 			*AFP_Buffer;
long 			AFP_Size = BUFFER_SIZE;
long 			AFP_Count = -1;
long 			AFP_Stride = -1;
int 			AFP_NumSectors = FLASH_NUM_SECTORS;
long 			AFP_SectorSize1 = FLASH_SECTOR_SIZE;
int 			AFP_SectorSize2 = FLASH_SECTOR_SIZE;
int 			AFP_Sector = -1;
int 			AFP_Error 			= 0;			// contains last error encountered
bool 			AFP_Verify 			= FALSE;		// verify writes or not
long 			AFP_StartOff 		= 0x0;			// sector start offset
long 			AFP_EndOff 			= 0x0;			// sector end offset
int				AFP_FlashWidth		= 0x10;			// width of the flash device
int 			*AFP_SectorInfo;
SECTORLOCATION SectorInfo[FLASH_NUM_SECTORS];



// flag telling us that the flash had to be reset
char reset = 0x0;

// exit flag
bool bExit = FALSE;


main()
{	
	int i = 0;
	// by making AFP_Buffer as big as possible the plug-in can send and
	// receive more data at a time making the data transfer quicker
	//
	// by allocating it on the heap the compiler does not create an
	// initialized array therefore making the driver image smaller
	// and faster to load
	//
	// we have modified the linker description file (LDF) so that the heap
	// is large enough to store BUFFER_SIZE elements at this point
	AFP_Buffer = (int *)malloc(BUFFER_SIZE);

	// AFP_Buffer will be NULL if we could not allocate storage for the
	// buffer
	if ( AFP_Buffer == NULL )
	{
		// tell GUI that our buffer was not initialized
		AFP_Error = BUFFER_IS_NULL;
	}

	//initiate sector information structures
	for(i=0;i<AFP_NumSectors; i++)
	{
		GetSectorStartEnd(&SectorInfo[i].lStartOff, &SectorInfo[i].lEndOff, i);
	}

	AFP_SectorInfo = (int*)&SectorInfo[0];

	// setup the flash so the DSP can access it
	SetupForFlash();
	
	// unlock all sectors
	Flash_UnlockAll ();
	
	//setup code so that flash programmer can just read memory instead of call GetCodes().
	GetCodes();

	// command processing loop
	while ( !bExit )
	{
		// the plug-in will set a breakpoint at "AFP_BreakReady" so it knows
		// when we are ready for a new command because the DSP will halt
		//
		// the jump is used so that the label will be part of the debug
		// information in the driver image otherwise it may be left out
		// since the label is not referenced anywhere
		asm("AFP_BreakReady:");
		if ( FALSE )
			asm("jump AFP_BreakReady;");

		// switch on the command
		switch ( AFP_Command )
		{
			// get manufacturer and device codes
			case GET_CODES:
				AFP_Error = GetCodes();
				break;

			// reset
			case RESET_FLASH:
				AFP_Error = ResetFlash();
				break;

			// write
			case WRITE:
				AFP_Error = WriteData( AFP_Offset, AFP_Count, AFP_Stride, AFP_Buffer );
//				AFP_Error = WriteData( 0x500, 0x600, 0x1, AFP_Buffer );
				break;

			// fill
			case FILL:
				AFP_Error = FillData( AFP_Offset, AFP_Count, AFP_Stride, AFP_Buffer );
				break;

			// erase all
			case ERASE_ALL:
				AFP_Error = EraseFlash();
				break;

			// erase sector
			case ERASE_SECT:
				if ((AFP_Sector >4) && (AFP_Sector % 4 != 0)) {
					break;
				}
				AFP_Error = EraseBlock( AFP_Sector );
				break;

			// read
			case READ:
				AFP_Error = ReadData( AFP_Offset, AFP_Count, AFP_Stride, AFP_Buffer );
				break;

			// get sector number based on address
			case GET_SECTNUM:
				AFP_Error = GetSectorNumber( AFP_Offset, &AFP_Sector );
				break;

			// get sector number start and end offset
			case GET_SECSTARTEND:
				AFP_Error = GetSectorStartEnd( &AFP_StartOff, &AFP_EndOff, AFP_Sector );
				break;

			// no command or unknown command do nothing
			case NO_COMMAND:
			default:
				// set our error
				AFP_Error = UNKNOWN_COMMAND;
				break;
		}

		// clear the command
		AFP_Command = NO_COMMAND;
	}

	// free the buffer if we were able to allocate one
	if ( AFP_Buffer )
		free( AFP_Buffer );


	// all done
	return 0;
}

void ami_setup (void) {

	*pEBIU_MODE = 0x1;		// bank 0 async access mode
	*pEBIU_AMGCTL = 0x3;

	// set port H MUX to configure PH8-PH13 as 1st Function (MUX = 00)
	// (bits 16-27 == 0) - Address signals A4-A9
	*pPORTH_FER = 0x3F00;
	*pPORTH_MUX = 0x0;
	*pPORTH_DIR_SET = 0x3F00;

	// configure PI0-PI15 as A10-A21 respectively
	*pPORTI_FER = 0xFFFF;
	*pPORTI_MUX = 0x0;
	*pPORTI_DIR_SET = 0xFFFF;

}

//////////////////////////////////////////////////////////////
// ERROR_CODE SetupForFlash()
//
// Perform necessary setup for the processor to talk to the
// flash such as external memory interface registers, etc.
//
//////////////////////////////////////////////////////////////

ERROR_CODE SetupForFlash()
{
	ami_setup();
	
	Flash_Setup ();	
	
	return NO_ERR;
}


//////////////////////////////////////////////////////////////
// ERROR_CODE WriteData()
//
// Write a buffer to flash.
//
// Inputs:	unsigned long ulStart - offset in flash to start the writes at
//			long lCount - number of elements to write, in this case bytes
//			long lStride - number of locations to skip between writes
//			int *pnData - pointer to data buffer
//
//////////////////////////////////////////////////////////////
ERROR_CODE WriteData( unsigned long ulStart, long lCount, long lStride, int *pnData )
{
	long i = 0;						// loop counter
	int j = 0;						// inner loop counter
	unsigned long ulOffset = ulStart;// current offset to write
	int iShift = 0;					// shift value by iShift bits
	int iNumWords = 2;				// number of words in a long
	int nLeftover = lCount % 4;		// how much if any do we have leftover to write
	ERROR_CODE ErrorCode = NO_ERR;	// tells whether there was an error trying to write
	unsigned short nCompare;				// value that we use to verify flash
	bool bVerifyError = FALSE;		// lets us know if there was a verify error
	int nSector = 0;

	//UnlockFlash(0);

	// if the user wants to verify then do it
	if( AFP_Verify == TRUE )
	{
		// write the buffer up to BUFFER_SIZE items
		for (i = 0; (i < lCount/4) && (i < BUFFER_SIZE); i++)
		{
			for (iShift = 0, j = 0; ( ( j < iNumWords ) && ( ErrorCode == NO_ERR ) ); j++, ulOffset += (lStride * 2))
			{
				// check to see that the address is within a valid sector
				ErrorCode = GetSectorNumber( ulOffset, &nSector );
				if(  ErrorCode == NO_ERR )		
				{
					// do the write, increase shift, and wait for completion
					ErrorCode = WriteFlash( ulOffset, (pnData[i] >> iShift) );
					//ErrorCode = PollToggleBit(ulOffset);
					ReadFlash( ulOffset, &nCompare );
					if( nCompare != (unsigned short)( (pnData[i]  >> iShift) & 0x0000FFFF ) )
					{
						bVerifyError = TRUE;
						break;
					}
					iShift += 16;
				}
				else
				{	// handle error
					bVerifyError = TRUE;
					return ErrorCode;
				}
			}
		}

		// because of the way our ldr file is built, we will always have
		// 2 bytes leftover if there is leftover, because the flash is 16 bit
		// that will mean only one write to do.
		if( ( nLeftover > 0 ) && ( ErrorCode == NO_ERR ) && bVerifyError == FALSE )
		{
			// check to see that the address is within a valid sector
			ErrorCode = GetSectorNumber( ulOffset, &nSector );
			if( NO_ERR == ErrorCode )
			{
				// do the write, increase shift, and wait for completion
				ErrorCode = WriteFlash( ulOffset, pnData[i] );
				//ErrorCode = PollToggleBit(ulOffset);
				ReadFlash( ulOffset, &nCompare );
				if( nCompare != (unsigned short)( pnData[i] & 0x0000FFFF ) )
				{
					bVerifyError = TRUE;
				}
			}
			else
			{
				return ErrorCode;
			}
		}

		// return appropriate error code if there was a verification error
		if( bVerifyError == TRUE )
			return VERIFY_WRITE;
	}
	// the user does not want to verify
	else
	{
		// write the buffer up to BUFFER_SIZE items
		for (i = 0; (i < lCount/4) && (i < BUFFER_SIZE); i++)
		{
			for (iShift = 0, j = 0; ( ( j < iNumWords ) && ( ErrorCode == NO_ERR ) ); j++, ulOffset += (lStride * 2))
			{
				// check to see that the address is within a valid sector
				ErrorCode = GetSectorNumber( ulOffset, &nSector );
				//ErrorCode = NO_ERR;
				if(  ErrorCode == NO_ERR )		
				{
					// do the write, increase shift, and wait for completion
					if ((ErrorCode = WriteFlash( ulOffset, (pnData[i] >> iShift) )) != NO_ERR) 
						return (ErrorCode);
					
					//ErrorCode = PollToggleBit(ulOffset);
					iShift += 16;
				}
				else
				{
					return ErrorCode;
				}
			}
		}

		// because of the way our ldr file is built, we will always have
		// 2 bytes leftover if there is leftover, because the flash is 16 bit
		// that will mean only one write to do.
		if( ( nLeftover > 0 ) && ( ErrorCode == NO_ERR ) )
		{
			// check to see that the address is within a valid sector
			ErrorCode = GetSectorNumber( ulOffset, &nSector );
			//ErrorCode = NO_ERR;
			if(  ErrorCode == NO_ERR )
			{
				// do the write, increase shift, and wait for completion
				ErrorCode = WriteFlash( ulOffset, pnData[i] );
				//ErrorCode = PollToggleBit(ulOffset);
			}
			else
			{
				return ErrorCode;
			}
		}

	}

	// return the appropriate error code
	return ErrorCode;
}


//////////////////////////////////////////////////////////////
// ERROR_CODE FillData()
//
// Fill flash with a value.
//
// Inputs:	unsigned long ulStart - offset in flash to start the writes at
//			long lCount - number of elements to write, in this case bytes
//			long lStride - number of locations to skip between writes
//			int *pnData - pointer to data buffer
//
//////////////////////////////////////////////////////////////

ERROR_CODE FillData( unsigned long ulStart, long lCount, long lStride, int *pnData )
{
	long i = 0;							// loop counter
	unsigned long ulOffset = ulStart;	// current offset to write
	ERROR_CODE ErrorCode = NO_ERR;		// tells whether we had an error while filling
	unsigned short nCompare = 0;					// value that we use to verify flash
	bool bVerifyError = FALSE;			// lets us know if there was a verify error
	unsigned long ulNewOffset = 0;		// used if we have an odd address
	int nSector = 0;

	//UnlockFlash(0);

	// if we have an odd offset we need to write a byte
	// to the first location and the last
	if(ulOffset%2 != 0)
	{
		// read the offset - 1 and OR in our value
		ulNewOffset = ulOffset - 1;
		ReadFlash( ulNewOffset, &nCompare );
		nCompare &= 0xFF00;
		nCompare |= (pnData[0] & 0x00FF);

		// do the write, and wait for completion
		WriteFlash( ulNewOffset, nCompare );
		ErrorCode = PollToggleBit(ulNewOffset);

		// move to the last offset
		ulNewOffset = (ulOffset - 1) + (lCount * (lStride * 2) );

		// read the value and OR in our value
		ReadFlash( ulNewOffset, &nCompare );
		nCompare &= 0x00FF;
		nCompare |= (pnData[0] & 0xFF00);

		// do the write, and wait for completion
		WriteFlash( ulNewOffset, nCompare );
		ErrorCode = PollToggleBit(ulNewOffset);

		// increment the offset and count
		ulOffset = ( (ulOffset - 1) + (lStride * 2) );
		lCount--;
	}

	// verify writes if the user wants to
	if( AFP_Verify == TRUE )
	{
		// fill the value
		for (i = 0; ( ( i < lCount ) && ( ErrorCode == NO_ERR ) ); i++, ulOffset += (lStride * 2) )
		{
			// check to see that the address is within a valid sector
			ErrorCode = GetSectorNumber( ulOffset, &nSector );
			if( NO_ERR == ErrorCode )
			{
				// do the write, and wait for completion
				WriteFlash( ulOffset, pnData[0] );
				ErrorCode = PollToggleBit(ulOffset);
				ReadFlash( ulOffset, &nCompare );
				if( nCompare != (unsigned short)( pnData[0] & 0x0000FFFF ) )
				{
					bVerifyError = TRUE;
					break;
				}
			}
			else
			{
				return ErrorCode;
			}

		}

		// return appropriate error code if there was a verification error
		if( bVerifyError == TRUE )
			return VERIFY_WRITE;
	}
	// user did not want to verify writes
	else
	{
		// fill the value
		for (i = 0; ( ( i < lCount ) && ( ErrorCode == NO_ERR ) ); i++, ulOffset += (lStride * 2) )
		{
			// check to see that the address is within a valid sector
			ErrorCode = GetSectorNumber( ulOffset, &nSector );
			if( NO_ERR == ErrorCode )
			{
				// do the write, and wait for completion
				ErrorCode = WriteFlash( ulOffset, pnData[0] );
				//ErrorCode = PollToggleBit(ulOffset);
			}
			else
			{
				return ErrorCode;
			}
		}
	}

	// return the appropriate error code
	return ErrorCode;
}


//////////////////////////////////////////////////////////////
// ERROR_CODE ReadData()
//
// Read a buffer from flash.
//
// Inputs:	unsigned long ulStart - offset in flash to start the reads at
//			int nCount - number of elements to read, in this case bytes
//			int nStride - number of locations to skip between reads
//			int *pnData - pointer to data buffer to fill
//
//////////////////////////////////////////////////////////////

ERROR_CODE ReadData( unsigned long ulStart, long lCount, long lStride, int *pnData )
{

	long i = 0;						// loop counter
	int j = 0;						// inner loop counter
	unsigned long ulOffset = ulStart;			// current offset to write
	int iShift = 0;					// shift value by iShift bits
	int iNumWords = 2;				// number of words in a long
	int nLeftover = lCount % 4;		// how much if any do we have leftover to write
	unsigned short nHi,nLow, nFlashData;
	int nSector = 0;
	ERROR_CODE ErrorCode = NO_ERR;		// tells whether we had an error while filling

	// write the buffer up to BUFFER_SIZE items
	for (i = 0; (i < lCount/4) && (i < BUFFER_SIZE); i++)
	{
		for ( iShift = 0, j = 0; j < iNumWords ; j+=2 )
		{

			// check to see that the address is within a valid sector
			ErrorCode = GetSectorNumber( ulOffset, &nSector );
			if( NO_ERR == ErrorCode )
			{
				// Read flash
				pnData[i] = 0;
				ReadFlash( ulOffset, &nLow );
				ulOffset += (lStride * 2);
				ReadFlash( ulOffset, &nHi );
				ulOffset += (lStride * 2);
				pnData[i] = (int)((nHi << 16) + nLow);
			}
			else
			{
				return ErrorCode;
			}
		}
	}

	// because of the way our ldr file is built, we will always have
	// 2 bytes leftover if there is leftover, because the flash is 16 bit
	// that will mean that there is only one write left to do.
	if( nLeftover > 0 )
	{

		// check to see that the address is within a valid sector
		ErrorCode = GetSectorNumber( ulOffset, &nSector );
		if( NO_ERR == ErrorCode )
		{
			ReadFlash( ulOffset, &nFlashData );
			pnData[i] = (int)nFlashData;
		}
		else
		{
			return ErrorCode;
		}
	}

	// return the appropriate error code
	return ErrorCode;
}


//////////////////////////////////////////////////////////////
// ERROR_CODE WriteFlash()
//
// Write a value to an offset in flash.
//
// Inputs:	unsigned long ulOffset - offset to write to
//			int nValue - value to write
//
//////////////////////////////////////////////////////////////

ERROR_CODE WriteFlash( unsigned long ulOffset, int nValue )
{
	unsigned short nStatus;
	
	if ((nStatus = Flash_ProgramWord(FLASH_START_ADDRESS, ulOffset, (unsigned short)nValue)) == 0x0080) return NO_ERR;

	return POLL_TIMEOUT;
}

//////////////////////////////////////////////////////////////
// ERROR_CODE ReadFlash()
//
// Read a value from an offset in flash.
//
// Inputs:	unsigned long ulOffset - offset to read from
//			int pnValue - pointer to store value read from flash
//
//////////////////////////////////////////////////////////////

ERROR_CODE ReadFlash( unsigned long ulOffset, unsigned short *pnValue )
{
	*pnValue = Flash_ReadWord (ulOffset);
	// ok
	return NO_ERR;
}


//////////////////////////////////////////////////////////////
// ERROR_CODE PollToggleBit()
//
// Polls the toggle bit in the flash to see when the operation
// is complete.
//
// Inputs:	unsigned long ulOffset - offset in flash
//
//////////////////////////////////////////////////////////////

ERROR_CODE PollToggleBit(unsigned long ulOffset)
{
	//This function has nothing to do in the current implementation.	
	return NO_ERR;
}


//////////////////////////////////////////////////////////////
// ERROR_CODE ResetFlash()
//
// Sends a "reset" command to the flash.
//
//////////////////////////////////////////////////////////////

ERROR_CODE ResetFlash()
{
	Flash_Reset ();
	
	// reset should be complete
	return NO_ERR;
}


//////////////////////////////////////////////////////////////
// ERROR_CODE EraseFlash()
//
// Sends an "erase all" command to the flash.
//
//////////////////////////////////////////////////////////////

ERROR_CODE EraseFlash()
{
	ERROR_CODE ErrorCode = NO_ERR;	// tells us if there was an error erasing flash
	unsigned short nError;

	nError = Flash_EraseChip ();
	if (nError == 0xff00) return POLL_TIMEOUT;

	// erase should be complete
	return NO_ERR;
}


//////////////////////////////////////////////////////////////
// ERROR_CODE EraseBlock()
//
// Sends an "erase block" command to the flash.
//
//////////////////////////////////////////////////////////////

ERROR_CODE EraseBlock( int nBlock )
{
	unsigned long ulSectorOff = 0x0;
	ERROR_CODE ErrorCode = NO_ERR;	// tells us if there was an error erasing flash
	unsigned short nStatus;


	// if the block is invalid just return
	if ( (nBlock < 0) || (nBlock > AFP_NumSectors) ) {
		return INVALID_BLOCK;
	} else {
		ulSectorOff = (nBlock * AFP_SectorSize1);
	}

	nStatus = Flash_EraseSector(FLASH_START_ADDRESS, ulSectorOff);
	if (nStatus == 0xff00) return (POLL_TIMEOUT);

	// block erase should be complete
	return NO_ERR;
}


//////////////////////////////////////////////////////////////
// ERROR_CODE UnlockFlash()
//
// Sends an "unlock" command to the flash to allow the flash
// to be programmed.
//
//////////////////////////////////////////////////////////////

ERROR_CODE UnlockFlash(unsigned long ulOffset)
{
	Flash_UnlockBlock (ulOffset);

	// ok
	return NO_ERR;
}


//////////////////////////////////////////////////////////////
// ERROR_CODE GetCodes()
//
// Sends an "auto select" command to the flash which will allow
// us to get the manufacturer and device codes.
//
//////////////////////////////////////////////////////////////

ERROR_CODE GetCodes()
{
	unsigned short nDevCode, nManuCode;
	
	Flash_ReadIdentifier (&nDevCode, &nManuCode);

	AFP_DevCode = nDevCode & 0x00ff;
	AFP_ManCode = nManuCode & 0x00ff;

	ResetFlash();

	// ok
	return NO_ERR;
}


//////////////////////////////////////////////////////////////
// ERROR_CODE GetSectorNumber()
//
// Gets a sector number based on the offset.
//
//////////////////////////////////////////////////////////////

ERROR_CODE GetSectorNumber( unsigned long ulOffset, int *pnSector )
{
	int nBlock;
	
	*pnSector = (int)((float)ulOffset / (float)AFP_SectorSize1);

	// ok
	return NO_ERR;
}

//////////////////////////////////////////////////////////////
// ERROR_CODE GetSectorStartEnd()
//
// Gets a sector number based on the offset.
//
// Inputs:	long *lStartOff - pointer to the start offset
//			long *lEndOff - pointer to the end offset
//			int nSector - sector number
//
//////////////////////////////////////////////////////////////

ERROR_CODE GetSectorStartEnd( long *lStartOff, long *lEndOff, int nSector )
{
	long lSectorSize = 0;

	*lStartOff = (long)(nSector * AFP_SectorSize1);
	*lEndOff = (long)(*lStartOff + AFP_SectorSize1 - 1);
	// ok
	return NO_ERR;
}
